/***********************************************************************
ESPRIT-Forest: Parallel Clustering of Massive Amplicon Sequence Data in Subquadratic Time 
by: Yunpeng Cai, Yijun Sun, Wei Zheng, Jin Yao and Yujie Yang  (C) 2016
Please kindly cite [Y.Cai et.al PLOS Comp. Biol. 2016]

THE LICENSED WORK IS PROVIDED UNDER THE TERMS OF THE ADAPTIVE PUBLIC LICENSE ("LICENSE") AS FIRST COMPLETED BY: _Yunpeng Cai, Yijun Sun, Wei Zheng, Jin Yao, Yujie Yang_ [Insert the name of the Initial Contributor here]. ANY USE, PUBLIC DISPLAY, PUBLIC PERFORMANCE, REPRODUCTION OR DISTRIBUTION OF, OR PREPARATION OF DERIVATIVE WORKS BASED ON, THE LICENSED WORK CONSTITUTES RECIPIENT'S ACCEPTANCE OF THIS LICENSE AND ITS TERMS, WHETHER OR NOT SUCH RECIPIENT READS THE TERMS OF THE LICENSE. "LICENSED WORK" AND "RECIPIENT" ARE DEFINED IN THE LICENSE. A COPY OF THE LICENSE IS LOCATED IN THE TEXT FILE ENTITLED "LICENSE.TXT" ACCOMPANYING THE CONTENTS OF THIS FILE. IF A COPY OF THE LICENSE DOES NOT ACCOMPANY THIS FILE, A COPY OF THE LICENSE MAY ALSO BE OBTAINED AT THE FOLLOWING WEB SITE: http://www.acsu.buffalo.edu/~yijunsun/lab/ESPRIT-Forest.html [Insert Initial Contributor's Designated Web Site here]

Software distributed under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for the specific language governing rights and limitations under the License.
*/

#include <iostream>
#include <fstream>
#include <time.h>
#include <math.h>
#include "global.h"
#include "kmer.h"
#include "TreeClust.h"
#include "needle.h"
#include "MinHeap.h"
using namespace std;

typedef struct{
	int idx;
	float dist;
}KDist;

int CompDist(const void * a, const void * b)
{
	KDist *kd1,*kd2;
	kd1=(KDist *)a;
	kd2=(KDist *)b;
	if (kd1->dist > kd2->dist) return 1;
	if (kd1->dist < kd2->dist) return -1;
	return 0;
}

TreeClust::TreeClust(float lmin,float lmax)
{
	root=NULL;
	levelinc=Global::level_inc;
	levelmax=lmax*levelinc;
	levelmin=levelmax;
	NumSeq=0;
	while (levelmin/levelinc > lmin) levelmin=levelmin/levelinc;
}

Tree* TreeClust::CreateBranch(int uid,float levelup)
{
	Tree *base=new Tree(uid);
	Tree *upper;
	float level=levelmin;
	while (level <=levelup*1.001)
	{
		upper=new Tree(uid);
		upper->AddChild(base);
		
		upper->SetThres(level);
		level*=levelinc;
		base=upper;
	}
	return base;
}	

Tree *TreeClust::Condense(Tree *node,int uid)
{
	Tree *newchild=CreateBranch(uid,node->GetThres()/levelinc);
	while (node->NumChildren() >0)
		node->DeleteChild(node->FirstChild());
	node->AddChild(newchild);
	while (!newchild->IsLeaf())
		newchild=newchild->FirstChild();
	return newchild;
}

Tree *TreeClust::AddSeq(int uid)
{
	
	Tree *current,*bestchild;
	float nrdist;
	NumSeq++;
	
	if (root==NULL)
	{
		root=CreateBranch(uid,levelmax*levelinc);	
		current=root;
		while (!current->IsLeaf()) 
		{
			current=current->FirstChild();
		}
		return current;
	}
	
	current=root;

	do{
		if (current->IsLeaf()) cerr <<"Shouldn't Get to Leaf" <<endl;
		
		if (current->BottomLevel())
		{
			bestchild=new Tree(uid);
			current->AddChild(bestchild);
			break;
		}
		
		if (current->NumChildren()==1)
		{
			bestchild=current->FirstChild();
			if (current==root)
			{
				nrdist=Kdist2Ndist(KmerDist(uid,bestchild->UID)); 
			}
		}
		else
		{
			nrdist=current->FindSpanChild(uid,bestchild);
		}
		
		if (bestchild !=NULL && nrdist < bestchild->GetThres())
		{
			current=bestchild;
		}
		else
		{
			bestchild=CreateBranch(uid,current->GetThres()/levelinc);
			current->AddChild(bestchild);
			break;
		}
	}while(1);
		
	while (!bestchild->IsLeaf()) 
	{
		bestchild=bestchild->FirstChild();
	}
	return bestchild;
}

Tree *TreeClust::AddSeqFrom(int uid, int branch, Tree *par)
{
	Tree *current,*bestchild;
	float nrdist;
	NumSeq++;
	
	current=par->FindChild(branch);
	if (current==NULL)
	{
		cerr <<" Wrong Branch ID " <<endl;
		return NULL;
	}
	
	nrdist=current->FindSpanChild(uid,bestchild);
	
	do{
		if (current->IsLeaf()) cerr <<"Shouldn't Get to Leaf" <<endl;
		
		if (current->BottomLevel())
		{
			bestchild=new Tree(uid);
			current->AddChild(bestchild);
			break;
		}
		
		if (bestchild !=NULL && nrdist < bestchild->GetThres())
		{
			current=bestchild;
			if (current->NumChildren()==1)
			{
				bestchild=current->FirstChild();
			}
			else
			{
				nrdist=current->FindSpanChild(uid,bestchild);
			}
		}
		else
		{
			bestchild=CreateBranch(uid,current->GetThres()/levelinc);
			current->AddChild(bestchild);
			break;
		}
	}while(1);
		
	while (!bestchild->IsLeaf()) 
	{
		bestchild=bestchild->FirstChild();
	}
	return bestchild;
}


Tree *TreeClust::AddSeqAt(int uid, Tree *par){
	Tree *bestchild;
	NumSeq++;
	
	if (par->BottomLevel())
	{
		bestchild=new Tree(uid);
		par->AddChild(bestchild);
	}
	else
	{
		bestchild=CreateBranch(uid,par->GetThres()/levelinc);
		par->AddChild(bestchild);
	}

	while (!bestchild->IsLeaf()) 
	{
		bestchild=bestchild->FirstChild();
	}
	return bestchild;
}


Tree* TreeClust::FindInsertParent(int uid)
{
	Tree *current,*bestchild;
	float nrdist;
	
	if (root==NULL)
	{
		return NULL;
	}
	
	current=root;

	do{
		if (current->BottomLevel() || current->IsLeaf())
		{
			return current;
		}
		
		if (current->NumChildren()==1)
		{
			bestchild=current->FirstChild();
			if (current==root)
			{
				nrdist=Kdist2Ndist(KmerDist(uid,bestchild->UID)); 
			}
		}
		else
		{
			nrdist=current->FindSpanChild(uid,bestchild);
		}
		
		if (bestchild !=NULL && nrdist < bestchild->GetThres())
		{
			current=bestchild;
		}
		else
		{
			return current;
		}
	}while(1);
		
	return NULL;
}

void TreeClust::RemoveNode(Tree *node)
{
	if (node==NULL)
	{
		fprintf(stderr,"Error Deleting Empty Node\n");
		exit(0);
	}
	Tree *ptr=node->GetParent();
	do{
		ptr->DeleteChild(node);
		node=ptr;
		ptr=node->GetParent();
	}while(node->IsLeaf() && ptr !=NULL); 
	if (node==root && node->IsLeaf())
	{
		root=NULL;
		delete node;
	}
}

void TreeClust::EstimateNN(int uid, ClustRec *clRec, SwapRec *tRec)
{
	Tree *top=clRec[uid].Node;
	Tree *par=top->GetParent();
	Tree *brother;
	
	while (par !=root && par->NumChildren() <=1)
	{
		top=par;
		par=top->GetParent();
	}
	if (par == root) return;
	int NNseq;
	float Ndist;
	if (par->UID !=uid && clRec[par->UID].clsid ==-1)
	{
		NNseq=par->UID;
	}
	else
	{
		Tree *brother=top->GetBrother();
		while (!brother->IsLeaf())
			brother=brother->FirstChild();
		NNseq=brother->UID;
	}

	Ndist=NeedleDist(uid,NNseq,par->GetThres());
	tRec->NNseq=NNseq;
	tRec->NNdist=Ndist;
}

void TreeClust::FindNN(int uid,SwapRec *tRec, float leveldn)
{
	vector<Tree *> tvec;
	
	MinHeap heap;
	float dist;
	Tree *top;
	
	if (root==NULL) return;
	
	tvec.clear();	
	root->ListChildren(tvec);
	for (int i=0; i<tvec.size();i++)
	{
		while (!tvec[i]->IsLeaf() && tvec[i]->NumChildren()==1)
		{
			tvec[i]=tvec[i]->FirstChild();
		}
		dist=KmerDist(tvec[i]->UID,uid); 
		if (dist < KdistBound(tRec->NNdist+tvec[i]->GetThres()))
		{
			heap.Add((void *)tvec[i],dist);
		}
	}
	
	//perform A* search
	while (!heap.Empty())
	{
		dist=heap.Pop((void *&)top);
		
		if (dist < KdistBound(tRec->NNdist+top->GetThres()))
		{
			if (top->IsLeaf()) //reach terminal
			{
				if (top->UID==uid) continue;
				dist=NeedleDist(top->UID,uid,min(levelmax,tRec->NNdist+0.02f));
				if (dist < tRec->NNdist)
				{
					tRec->NNdist2=tRec->NNdist;
					tRec->NNseq2=tRec->NNseq;
					tRec->NNdist=dist;
					tRec->NNseq=top->UID;
					if (dist <leveldn)
						return;
				}
				else if (dist < tRec->NNdist2)
				{
					tRec->NNdist2=dist;
					tRec->NNseq2=top->UID;
				}
			}
			else //expand
			{
				while (!top->BottomLevel() && top->NumChildren()==1)
					top=top->FirstChild();
				tvec.clear();	
				top->ListChildren(tvec);

				for (int i=0; i<tvec.size();i++)
				{
					dist=KmerDist(tvec[i]->UID,uid); 
					if (dist < KdistBound(tRec->NNdist+tvec[i]->GetThres()))
					{
						heap.Add((void *)tvec[i],dist);
					}
				}
			}
		}	
	}
}

void TreeClust::ListLeaves(FILE *fp)
{
	vector<Tree *> tvec;
	tvec.clear();	
	root->ListLeaf(tvec);
	for (int i=0; i<tvec.size();i++)
	{	
		fprintf(fp,"[%d]",tvec[i]->UID);
	}
	fprintf(fp,"\n");
	fflush(fp);
}

void TreeClust::ListBottom(vector<Tree *> &tvec)
{
	tvec.clear();
	root->ListBottom(tvec);
}
