/***********************************************************************
ESPRIT-Forest: Parallel Clustering of Massive Amplicon Sequence Data in Subquadratic Time 
by: Yunpeng Cai, Yijun Sun, Wei Zheng, Jin Yao and Yujie Yang  (C) 2016
Please kindly cite [Y.Cai et.al PLOS Comp. Biol. 2016]

THE LICENSED WORK IS PROVIDED UNDER THE TERMS OF THE ADAPTIVE PUBLIC LICENSE ("LICENSE") AS FIRST COMPLETED BY: _Yunpeng Cai, Yijun Sun, Wei Zheng, Jin Yao, Yujie Yang_ [Insert the name of the Initial Contributor here]. ANY USE, PUBLIC DISPLAY, PUBLIC PERFORMANCE, REPRODUCTION OR DISTRIBUTION OF, OR PREPARATION OF DERIVATIVE WORKS BASED ON, THE LICENSED WORK CONSTITUTES RECIPIENT'S ACCEPTANCE OF THIS LICENSE AND ITS TERMS, WHETHER OR NOT SUCH RECIPIENT READS THE TERMS OF THE LICENSE. "LICENSED WORK" AND "RECIPIENT" ARE DEFINED IN THE LICENSE. A COPY OF THE LICENSE IS LOCATED IN THE TEXT FILE ENTITLED "LICENSE.TXT" ACCOMPANYING THE CONTENTS OF THIS FILE. IF A COPY OF THE LICENSE DOES NOT ACCOMPANY THIS FILE, A COPY OF THE LICENSE MAY ALSO BE OBTAINED AT THE FOLLOWING WEB SITE: http://www.acsu.buffalo.edu/~yijunsun/lab/ESPRIT-Forest.html [Insert Initial Contributor's Designated Web Site here]

Software distributed under the License is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for the specific language governing rights and limitations under the License.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <iostream>
#include "needle.h"

using namespace std;

#define MinFloat -1e30
extern void* Malloc(size_t size);

DIRS GetMaxInc(float *Vt, float ia, float ib, float ic, float &newval)
{
	DIRS dirs=DIR_DIAG;
	newval=Vt[DIR_DIAG]+ia;
	if (Vt[DIR_LEFT] + ib > newval)
		{
			newval=Vt[DIR_LEFT] + ib;
			dirs=DIR_LEFT;
		}	
	if (Vt[DIR_UP] + ic > newval)
		{
			newval=Vt[DIR_UP] + ia;
			dirs=DIR_UP;
		}	
	return dirs;
}

float Needle::Score(int row, int col)
{
    if (row==4 && col==4) return 0;
    if (row==col)
    	return 5;
   	return -4;
}

float Needle::Score(int row, ProbChar &p2)
{
	if (row>4|| row <=-1) return 0;
	if (row==-1) cerr << "here "<<endl;
	if (row==4) return -4*(1-p2[0]-p2[Num_RNAType+1]-p2[Num_RNAType+2]);
	return 9*p2[row+1]-4*(1-p2[0]-p2[Num_RNAType+2]);
}

float Needle::Score(ProbChar &p1, ProbChar &p2)
{
	float ret=0.0;
	for (int i=1;i<=Num_RNAType;i++)
	{
		ret+=p1[i]*(9*p2[i]-4*(1-p2[0]-p2[Num_RNAType+2]));
	}
	ret-=4*p1[Num_RNAType+1]*(1-p2[0]-p2[Num_RNAType+1]-p2[Num_RNAType+2]);
	return ret;
}

int* Needle::StrToCode(char* str)
{
	int len=strlen(str);
	int *l=(int *)Malloc(strlen(str)*sizeof(int));
	
	for (int i = 0; i < len; i++)
	{
		switch(toupper(str[i]))
		{
		case 'A':
				l[i]=0;
				break;
		case 'G':
				l[i]=1;
				break;
		case 'C':
				l[i]=2;
				break;
		case 'T':
				l[i]=3;
				break;
		case 'N':
			  l[i]=4;
			  break;		
		default:
				l[i]=-1;
				break;
		}
	}
	return l;
}

float Needle::CalculateMatrix_Diag(int *source, int *dest, int slen, int dlen, int diag_dn, int diag_up, DIRS *pr, DIRS &defVal)
{
	int i,j;
	float inc,kdist;

	float *ar;
	float lefttrack[NumDIRS],thistrack[NumDIRS];
	DIRS dirs;
	
	 
	int rdif=max(dlen-slen,0);
	int ldif=max(slen-dlen,0);
 	float lastv;
	
	ar= (float *)Malloc(NumDIRS*(dlen+1)*sizeof(float));
	
 	ar[DIR_DIAG]=0;
 	ar[DIR_LEFT]=MinFloat;
 	ar[DIR_UP]=MinFloat;
 	
 	lastv=0;
	for (i = 1; i <= dlen; i++)
	{
  		ar[i*NumDIRS+DIR_DIAG]=MinFloat;
  		ar[i*NumDIRS+DIR_UP]=MinFloat;
		if (i<=rdif+diag_up)
		{
			ar[i*NumDIRS+DIR_LEFT]=lastv;
			if (i<dlen)
				lastv+=gap_len;
		}
		else
		{
			ar[i*NumDIRS+DIR_LEFT]=MinFloat;
		}
	}
	
 	lastv=0;//gap_open;
        
	for (i = 1; i <= slen; i++)
	{
		lefttrack[DIR_DIAG]=MinFloat;
		lefttrack[DIR_LEFT]=MinFloat;
		if (i<=ldif+diag_dn)
		{
			lefttrack[DIR_UP]=lastv;
			if (i<slen)
				lastv+=gap_len;
		}
		else
		{
			lefttrack[DIR_UP]=MinFloat;
		}
		int colstart=max(1,i-ldif-diag_dn);
 		int colend=min(dlen,i+rdif+diag_up);
		for (j = colstart; j <= colend; j++)
		{  
  			dirs=GetMaxInc(ar+NumDIRS*(j-1),0,0,0,kdist);
			kdist+=Score(source[i-1],dest[j-1]);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_DIAG]=dirs;
			thistrack[DIR_DIAG]=kdist;
			if (i==slen)
				dirs=GetMaxInc(lefttrack,gap_len,gap_len,gap_len,kdist);
			else
				dirs=GetMaxInc(lefttrack,gap_open,gap_len,gap_open,kdist);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_LEFT]=dirs;
			thistrack[DIR_LEFT]=kdist;
			if (j==dlen)
				dirs=GetMaxInc(ar+NumDIRS*j,gap_len,gap_len,gap_len,kdist);
			else
				dirs=GetMaxInc(ar+NumDIRS*j,gap_open,gap_open,gap_len,kdist);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_UP]=dirs;
			thistrack[DIR_UP]=kdist;
			for (int itr=0;itr<3;itr++)
			{
				ar[NumDIRS*(j-1)+itr]=lefttrack[itr];
				lefttrack[itr]=thistrack[itr];
			}
		}
		for (int itr=0;itr<3;itr++)
		{
			ar[NumDIRS*colend+itr]=thistrack[itr];  
		}
	}
  
	dirs=GetMaxInc(ar+NumDIRS*dlen,0,0,0,kdist);
	defVal=dirs;
	
	free(ar);
	return kdist;
}

float Needle::CalculateMatrix_Diag(int *source, ProbString &dest, int slen, int dlen, int diag_dn, int diag_up,  DIRS *pr, DIRS &defVal)
{
	int i,j;
	float inc,kdist;

	float *ar;
	float lefttrack[NumDIRS],thistrack[NumDIRS];
	DIRS dirs;
	
	 
	int rdif=max(dlen-slen,0);
	int ldif=max(slen-dlen,0);
 	float lastv;
	
	ar= (float *)Malloc(NumDIRS*(dlen+1)*sizeof(float));
	
 	ar[DIR_DIAG]=0;
 	ar[DIR_LEFT]=MinFloat;
 	ar[DIR_UP]=MinFloat;
 	
 	lastv=0;

	for (i = 1; i <= dlen; i++)
	{
  		ar[i*NumDIRS+DIR_DIAG]=MinFloat;
  		ar[i*NumDIRS+DIR_UP]=MinFloat;
		if (i<=rdif+diag_up)
		{
			ar[i*NumDIRS+DIR_LEFT]=lastv;
			if (i<dlen)
    		lastv+=gap_len*(1-dest[i][0]);
		}
		else
		{
			ar[i*NumDIRS+DIR_LEFT]=MinFloat;
		}
	}
        
 	lastv=0; //gap_open;
        
	for (i = 1; i <= slen; i++)
	{
		lefttrack[DIR_DIAG]=MinFloat;
		lefttrack[DIR_LEFT]=MinFloat;
		if (i<=ldif+diag_dn)
		{
			lefttrack[DIR_UP]=lastv;
			if (i<slen)
				lastv+=gap_len;
		}
		else
		{
			lefttrack[DIR_UP]=MinFloat;
 		}
 		int colstart=max(1,i-ldif-diag_dn);
 		int colend=min(dlen,i+rdif+diag_up);
		for (j = colstart; j <= colend; j++)
		{  
			dirs=GetMaxInc(ar+NumDIRS*(j-1),0,0,0,kdist);
			kdist+=Score(source[i-1],dest[j-1]);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_DIAG]=dirs;
			thistrack[DIR_DIAG]=kdist;
			if (i==slen)
				dirs=GetMaxInc(lefttrack,gap_len*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),kdist);
			else
				dirs=GetMaxInc(lefttrack,gap_open*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),gap_open*(1-dest[j-1][0]),kdist);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_LEFT]=dirs;
			thistrack[DIR_LEFT]=kdist;
			if (j==dlen)
				dirs=GetMaxInc(ar+NumDIRS*j,gap_len,gap_len,gap_len,kdist);
			else
				dirs=GetMaxInc(ar+NumDIRS*j,gap_open,gap_open,gap_len,kdist);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_UP]=dirs;
			thistrack[DIR_UP]=kdist;
			for (int itr=0;itr<3;itr++)
			{
				ar[NumDIRS*(j-1)+itr]=lefttrack[itr];
				lefttrack[itr]=thistrack[itr];
			}
		}
		for (int itr=0;itr<3;itr++)
		{
			ar[NumDIRS*colend+itr]=thistrack[itr];  
		}
	}
  
	dirs=GetMaxInc(ar+NumDIRS*dlen,0,0,0,kdist);
	defVal=dirs;
	
	free(ar);
	return kdist;
}

float Needle::CalculateMatrix_Diag(ProbString &source, ProbString &dest, int slen, int dlen, int diag_dn, int diag_up, DIRS *pr, DIRS &defVal)
{
  int i,j;
  float inc,kdist;

	float *ar;
	float lefttrack[NumDIRS],thistrack[NumDIRS];
	DIRS dirs;
	
	 
	int rdif=max(dlen-slen,0);
	int ldif=max(slen-dlen,0);
 	float lastv;
	
	ar= (float *)Malloc(NumDIRS*(dlen+1)*sizeof(float));
	
 	ar[DIR_DIAG]=0;
 	ar[DIR_LEFT]=MinFloat;
 	ar[DIR_UP]=MinFloat;
 	
 	lastv=0;

	for (i = 1; i <= dlen; i++)
	{
  		ar[i*NumDIRS+DIR_DIAG]=MinFloat;
  		ar[i*NumDIRS+DIR_UP]=MinFloat;
		if (i<=rdif+diag_up)
		{
			ar[i*NumDIRS+DIR_LEFT]=lastv;
			if (i<dlen)
				lastv+=gap_len*(1-dest[i][0]);
		}
		else
		{
			ar[i*NumDIRS+DIR_LEFT]=MinFloat;
		}
	}
        
 	lastv=0; //gap_open*(1-source[0][0]);
        
	for (i = 1; i <= slen; i++)
	{
		lefttrack[DIR_DIAG]=MinFloat;
		lefttrack[DIR_LEFT]=MinFloat;
		if (i<=ldif+diag_dn)
		{
			lefttrack[DIR_UP]=lastv;
			if (i<slen)
				lastv+=gap_len*(1-source[i][0]);
		}
		else
		{
			lefttrack[DIR_UP]=MinFloat;
		}
 		int colstart=max(1,i-ldif-diag_dn);
 		int colend=min(dlen,i+rdif+diag_up);
		for (j = colstart; j <= colend; j++)
		{  
  			dirs=GetMaxInc(ar+NumDIRS*(j-1),0,0,0,kdist);
			kdist+=Score(source[i-1],dest[j-1]);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_DIAG]=dirs;
			thistrack[DIR_DIAG]=kdist;
			if (i==slen)						
				dirs=GetMaxInc(lefttrack,gap_len*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),kdist);
			else					
				dirs=GetMaxInc(lefttrack,gap_open*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),gap_open*(1-dest[j-1][0]),kdist);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_LEFT]=dirs;
			thistrack[DIR_LEFT]=kdist;
			if (j==dlen)					
				dirs=GetMaxInc(ar+NumDIRS*j,gap_len*(1-source[i-1][0]),gap_len*(1-source[i-1][0]),gap_len*(1-source[i-1][0]),kdist);
			else					
				dirs=GetMaxInc(ar+NumDIRS*j,gap_open*(1-source[i-1][0]),gap_open*(1-source[i-1][0]),gap_len*(1-source[i-1][0]),kdist);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_UP]=dirs;
			thistrack[DIR_UP]=kdist;
			for (int itr=0;itr<3;itr++)
			{
				ar[NumDIRS*(j-1)+itr]=lefttrack[itr];
				lefttrack[itr]=thistrack[itr];
			}
		}
		for (int itr=0;itr<3;itr++)
		{
			ar[NumDIRS*colend+itr]=thistrack[itr];  
		}
	}
  
	dirs=GetMaxInc(ar+NumDIRS*dlen,0,0,0,kdist);
	defVal=dirs;
	
	free(ar);
	return kdist;
}

float Needle::CalculateMatrix_NoDgap(int *source, ProbString &dest, int slen, int dlen, DIRS *pr, DIRS &defVal)
{
	int i,j;
	float inc,kdist;
	if (dlen < slen)
		cerr <<" Error Template len < string len " <<endl;

	float *ar;
	float lefttrack[NumDIRS],thistrack[NumDIRS];
	DIRS dirs;
	 
	ar= (float *)Malloc(NumDIRS*(dlen+1)*sizeof(float));
	
 	ar[DIR_DIAG]=0;
 	ar[DIR_LEFT]=MinFloat;
 	ar[DIR_UP]=MinFloat;
 	
 	float lastv;
	
 	lastv=0; //gap_open*(1-dest[0][0]);
	for (i = 1; i <= dlen; i++)
	{
  		ar[i*NumDIRS+DIR_DIAG]=MinFloat;
  		ar[i*NumDIRS+DIR_UP]=MinFloat;
		ar[i*NumDIRS+DIR_LEFT]=MinFloat;
	}
        
 	lastv=0; //gap_open;
        
	for (i = 1; i <= slen; i++)
	{
		lefttrack[DIR_DIAG]=MinFloat;
		lefttrack[DIR_LEFT]=MinFloat;
		lefttrack[DIR_UP]=MinFloat;
		for (j = i; j <= i+(dlen-slen); j++)
		{  
			dirs=GetMaxInc(ar+NumDIRS*(j-1),0,0,0,kdist);
			kdist+=Score(source[i-1],dest[j-1]);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_DIAG]=dirs;
			thistrack[DIR_DIAG]=kdist;
			if (i==slen)					
				dirs=GetMaxInc(lefttrack,gap_len*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),kdist);
			else				
				dirs=GetMaxInc(lefttrack,gap_open*(1-dest[j-1][0]),gap_len*(1-dest[j-1][0]),gap_open*(1-dest[j-1][0]),kdist);
			pr[NumDIRS*(i*(dlen+1)+j)+DIR_LEFT]=dirs;
			thistrack[DIR_LEFT]=kdist;
			thistrack[DIR_UP]=MinFloat;
			for (int itr=0;itr<3;itr++)
			{
				ar[NumDIRS*(j-1)+itr]=lefttrack[itr];
				lefttrack[itr]=thistrack[itr];
			}
		}	
		for (int itr=0;itr<3;itr++)
		{
			ar[NumDIRS*(i+(dlen-slen))+itr]=thistrack[itr];  
		}
	}
  
	dirs=GetMaxInc(ar+NumDIRS*dlen,0,0,0,kdist);
	defVal=dirs;
	
	free(ar);
	return kdist;
}

void Needle::GetAlignments(DIRS *pr, DIRS defVal, char *sA, char *sB, char *alA,char *alB)
{
	char *buf,*ptrA,*ptrB,*pbuf;
	int i,j;
	DIRS endpt,newpt;
	
  ptrA=alA;
  ptrB=alB;       
         
  i=strlen(sA);
  j=strlen(sB);    
  int dlen=j;

  buf=(char *)Malloc((i+j+1)*sizeof(char));
  
  endpt=defVal;
  
  while (i > 0 && j > 0)
  {
    newpt=pr[NumDIRS*(i*(dlen+1)+j)+endpt];
    if (endpt==DIR_DIAG)
       {
         *(ptrA++) = sA[i-1];
         *(ptrB++) = sB[j-1];
         i--;j--;                
       }
    else if (endpt==DIR_UP)
      {
        *(ptrA++) = sA[i-1];
        *(ptrB++) = '-';
        i--;
      }
    else if (endpt==DIR_LEFT)
      {
        *(ptrA++) = '-';
        *(ptrB++) = sB[j-1];
        j--;
      }
    else
    {	
    	fprintf(stderr,"Error No Match %d %d\n",i,j);
    	exit(0);
    }  
    endpt=newpt;
  }
  while(i > 0)
  {
    *(ptrA++) = sA[i-1];
    *(ptrB++) = '-';
    i--;            
  }
  while(j > 0)
  {
    *(ptrA++) = '-';
    *(ptrB++) = sB[j - 1];
    j--;            
  }
	ptrA--;ptrB--;

	pbuf=buf;
	do
	{
		*(pbuf++)=*(ptrA--);
	}while (ptrA >=alA);
	*(pbuf)=0;
	strcpy(alA,buf);
	  
	pbuf=buf;
	do
	{
		*(pbuf++)=*(ptrB--);
	}while (ptrB >=alB);
	*(pbuf)=0;
	strcpy(alB,buf);
	free (buf);
}

void Needle::GetAlignments(DIRS *pr, DIRS defVal, ProbString &sA, ProbString &sB, ProbString &alA,ProbString &alB)
{
	ProbChar *bufA,*bufB,*ptrA,*ptrB;
	int i,j;
	DIRS endpt,newpt;

	i=sA.Len();
	j=sB.Len();          
	
	int dlen=j;
	
	bufA=new ProbChar[i+j];
	bufB=new ProbChar[i+j];
	
	ptrA=bufA;
	ptrB=bufB;      

	endpt=defVal;
       
	while (i > 0 && j > 0)
	{
		newpt=pr[NumDIRS*(i*(dlen+1)+j)+endpt];
		if (endpt==DIR_DIAG)
		{
			*(ptrA++) = sA[i-1];
			*(ptrB++) = sB[j-1];
			i--;j--;                
		}
		else if (endpt==DIR_UP)
		{
			*(ptrA++) = sA[i-1];
			*(ptrB++) = gapChar;
			i--;
		}
		else if (endpt==DIR_LEFT)
		{
			*(ptrA++) = gapChar;
			*(ptrB++) = sB[j-1];
			j--;
		}
		else
		{	
			cerr <<"Error No Match" << i << " " << j << endl;
			cerr <<sA.ToString()<<endl;
			cerr <<sB.ToString()<<endl;
			exit(0);
		}  
		endpt=newpt;
	}
	while(i > 0)
	{
		*(ptrA++) = sA[i-1];
		*(ptrB++) = gapChar;
		i--;            
	}
	while(j > 0)
	{
		*(ptrA++) = gapChar;
		*(ptrB++) = sB[j - 1];
		j--;            
	}
	
	
	alA.Alloc((int) (ptrA-bufA));
	alB.Alloc((int) (ptrB-bufB));
	
	for (i=0;i<alA.Len();i++) alA[i]=*(--ptrA);
	for (i=0;i<alB.Len();i++) alB[i]=*(--ptrB);
	if (ptrA!=bufA || ptrB !=bufB)
	{
		cerr << "Needle align ptr error\n"<<endl;
	}
	//convert gap to end-gap
	int ptr;
	
	i=0;
	while (i <alA.Len() && alA[i].isGap()) alA[i++]=EndgapChar;
	j=alA.Len()-1;
	while (j>=0 && alA[j].isGap()) 	alA[j--]=EndgapChar;
	for (ptr=i+1;ptr<j;ptr++)
	{
		if (alA[ptr].isGap())
		{
			alA[ptr][Num_RNAType+2]=max(alA[ptr-1][Num_RNAType+2],alA[ptr][Num_RNAType+2]);
			alA[ptr][0]=1.0-alA[ptr][Num_RNAType+2];
		}
	}
	for (ptr=j-1;ptr>i;ptr--)
	{
		if (alA[ptr].isGap())
		{
			alA[ptr][Num_RNAType+2]=max(alA[ptr+1][Num_RNAType+2],alA[ptr][Num_RNAType+2]);
			alA[ptr][0]=1.0-alA[ptr][Num_RNAType+2];
		}
	}
	
	i=0;
	while (i <alB.Len() && alB[i].isGap()) alB[i++]=EndgapChar;
	j=alB.Len()-1;
	while (j>=0 && alB[j].isGap()) alB[j--]=EndgapChar;
	for (ptr=i+1;ptr<j;ptr++)
	{
		if (alB[ptr].isGap())
		{
			alB[ptr][Num_RNAType+2]=max(alB[ptr-1][Num_RNAType+2],alB[ptr][Num_RNAType+2]);
			alB[ptr][0]=1.0-alB[ptr][Num_RNAType+2];
		}
	}
	for (ptr=j-1;ptr>i;ptr--)
	{
		if (alB[ptr].isGap())
		{
			alB[ptr][Num_RNAType+2]=max(alB[ptr+1][Num_RNAType+2],alB[ptr][Num_RNAType+2]);
			alB[ptr][0]=1.0-alB[ptr][Num_RNAType+2];
		}
	}
	
	delete [] bufA;
	delete [] bufB;
}

void Needle::GetAlignments(DIRS *pr, DIRS defVal, char *sA, ProbString &sB, char *alA,ProbString &alB)
{
	char *ptrA, *bufA;
	ProbChar *bufB,*ptrB;
	int i,j;
	DIRS endpt,newpt;

  i=strlen(sA);
  j=sB.Len();          
	
	int dlen=j;
	
	bufA=(char *)Malloc((i+j+1)*sizeof(char));
	bufB=new ProbChar[i+j];
	
	ptrA=bufA;
	ptrB=bufB;  

	endpt=defVal;  
       
	while (i > 0 && j > 0)
	{
		newpt=pr[NumDIRS*(i*(dlen+1)+j)+endpt];
		if (endpt==DIR_DIAG)
		{
			*(ptrA++) = sA[i-1];
			*(ptrB++) = sB[j-1];
			i--;j--;                
		}
		else if (endpt==DIR_UP)
		{
			*(ptrA++) = sA[i-1];
			*(ptrB++) = gapChar;
			i--;
		}
		else if (endpt==DIR_LEFT)
		{
			*(ptrA++) = '-';
			*(ptrB++) = sB[j-1];
			j--;
		}
		else
		{	
			cerr <<"Error No Match" << i << " " << j << endl;
			cerr <<sA<<endl;
			cerr <<sB.ToString()<<endl;
			exit(0);
		}  
		endpt=newpt;
	}
	while(i > 0)
	{
		*(ptrA++) = sA[i-1];
		*(ptrB++) = gapChar;
		i--;            
	}
	while(j > 0)
	{
		*(ptrA++) = '-';
		*(ptrB++) = sB[j - 1];
		j--;            
	}
		
	alB.Alloc((int) (ptrB-bufB));
	
	do {
		*(alA++)=*(--ptrA);
	} while (ptrA > bufA);		
	*alA=0;
	
	for (i=0;i<alB.Len();i++) alB[i]=*(--ptrB);

	int ptr;
	i=0;
	while (i <alB.Len() && alB[i].isGap()) alB[i++]=EndgapChar;
	j=alB.Len()-1;
	while (j>=0 && alB[j].isGap()) alB[j--]=EndgapChar;
	for (ptr=i+1;ptr<j;ptr++)
	{
		if (alB[ptr].isGap())
		{
			alB[ptr][Num_RNAType+2]=max(alB[ptr-1][Num_RNAType+2],alB[ptr][Num_RNAType+2]);
			alB[ptr][0]=1.0-alB[ptr][Num_RNAType+2];
		}
	}
	for (ptr=j-1;ptr>i;ptr--)
	{
		if (alB[ptr].isGap())
		{
			alB[ptr][Num_RNAType+2]=max(alB[ptr+1][Num_RNAType+2],alB[ptr][Num_RNAType+2]);
			alB[ptr][0]=1.0-alB[ptr][Num_RNAType+2];
		}
	}

	free(bufA);
	delete [] bufB;
}


void Needle::GetAlignments(DIRS *pr, DIRS defVal, ProbString &sA, ProbString &sB, ProbString &alA,ProbString &alB, int *posA, int *posB)
{
	ProbChar *bufA,*bufB,*ptrA,*ptrB;
	int i,j,pt;
	DIRS endpt,newpt;
	int count=0;
	
	i=sA.Len();
	j=sB.Len();          
	
	int dlen=j;
	
	bufA=new ProbChar[i+j];
	bufB=new ProbChar[i+j];
	
	ptrA=bufA;
	ptrB=bufB;       
       
	endpt=defVal; 
	
	while (i > 0 && j > 0)
	{
		newpt=pr[NumDIRS*(i*(dlen+1)+j)+endpt];
		if (endpt==DIR_DIAG)
		{
			*(ptrA++) = sA[i-1];
			*(ptrB++) = sB[j-1];
			i--;j--;
			posA[i]=count;
			posB[j]=count;
		}
		else if (endpt==DIR_UP)
		{
			*(ptrA++) = sA[i-1];
			*(ptrB++) = gapChar;
			i--;
			posA[i]=count;
		}
		else if (endpt==DIR_LEFT)
		{
			*(ptrA++) = gapChar;
			*(ptrB++) = sB[j-1];
			j--;
			posB[j]=count;
		}
		else
		{	
			cerr <<"Error No Match" << i << " " << j << endl;
			cerr <<sA.ToString()<<endl;
			cerr <<sB.ToString()<<endl;
			exit(0);
		}  
		endpt=newpt;
		count++;
	}
	while(i > 0)
	{
		*(ptrA++) = sA[i-1];
		*(ptrB++) = gapChar;
		i--;    
		posA[i]=count++;
	}
	while(j > 0)
	{
		*(ptrA++) = gapChar;
		*(ptrB++) = sB[j - 1];
		j--;            
		posB[j]=count++;
	}
	
	
	alA.Alloc((int) (ptrA-bufA));
	alB.Alloc((int) (ptrB-bufB));
	
	for (i=0;i<alA.Len();i++) 
	{
		alA[i]=*(--ptrA);
	}
	for (i=0;i<sA.Len();i++)
		posA[i]=alA.Len()-1-posA[i];
	/*	cerr << "printing out posA " <<endl;
	for (i=0;i< sA.Len();i++)
		cerr << posA[i] << " ";
	cerr <<endl;*/
	
	for (i=0;i<alB.Len();i++) 
	{
		alB[i]=*(--ptrB);
	}
	for (i=0;i<sB.Len();i++)
		posB[i]=alB.Len()-1-posB[i];
/*	cerr << "printing out posB " <<endl;
	for (i=0;i< sB.Len();i++)
		cerr << posB[i] << " ";
	cerr <<endl;*/
	
	if (ptrA!=bufA || ptrB !=bufB)
	{
		cerr << "Needle align ptr error\n"<<endl;
	}
	//convert gap to end-gap
	int ptr;
	
	i=0;
	while (i <alA.Len() && alA[i].isGap()) alA[i++]=EndgapChar;
	j=alA.Len()-1;
	while (j>=0 && alA[j].isGap()) 	alA[j--]=EndgapChar;
	for (ptr=i+1;ptr<j;ptr++)
	{
		if (alA[ptr].isGap())
		{
			alA[ptr][Num_RNAType+2]=max(alA[ptr-1][Num_RNAType+2],alA[ptr][Num_RNAType+2]);
			alA[ptr][0]=1.0-alA[ptr][Num_RNAType+2];
		}
	}
	for (ptr=j-1;ptr>i;ptr--)
	{
		if (alA[ptr].isGap())
		{
			alA[ptr][Num_RNAType+2]=max(alA[ptr+1][Num_RNAType+2],alA[ptr][Num_RNAType+2]);
			alA[ptr][0]=1.0-alA[ptr][Num_RNAType+2];
		}
	}
	
	i=0;
	while (i <alB.Len() && alB[i].isGap()) alB[i++]=EndgapChar;
	j=alB.Len()-1;
	while (j>=0 && alB[j].isGap()) alB[j--]=EndgapChar;
	for (ptr=i+1;ptr<j;ptr++)
	{
		if (alB[ptr].isGap())
		{
			alB[ptr][Num_RNAType+2]=max(alB[ptr-1][Num_RNAType+2],alB[ptr][Num_RNAType+2]);
			alB[ptr][0]=1.0-alB[ptr][Num_RNAType+2];
		}
	}
	for (ptr=j-1;ptr>i;ptr--)
	{
		if (alB[ptr].isGap())
		{
			alB[ptr][Num_RNAType+2]=max(alB[ptr+1][Num_RNAType+2],alB[ptr][Num_RNAType+2]);
			alB[ptr][0]=1.0-alB[ptr][Num_RNAType+2];
		}
	}
	
	delete [] bufA;
	delete [] bufB;
}

void Needle::Align(char *seq1,char *seq2,char *al1,char *al2,float diag_fact)
{
	DIRS *pr;
	DIRS entryVal;
	float score;
	int len1,len2;
	int *code1, *code2;
	int i,j;
	
	len1=strlen(seq1);
	len2=strlen(seq2);

	code1=StrToCode(seq1);
	code2=StrToCode(seq2);
	
	pr=(DIRS *)Malloc(NumDIRS*(len1+1)*(len2+1)*sizeof(DIRS));
	

	score=CalculateMatrix_Diag(code1, code2, len1, len2, (int) (diag_fact*max(len1,len2))+1, (int) (diag_fact*max(len1,len2))+1, pr,entryVal);
	GetAlignments(pr, entryVal, seq1, seq2, al1, al2);

	free(pr);
	free(code1);
	free(code2);
}

void Needle::Align(ProbString &str1,ProbString &str2,ProbString &al1, ProbString &al2,float diag_fact)
{
	DIRS *pr;
	DIRS entryVal;
	float score;
	int len1,len2;
	int i,j;
	
	len1=str1.Len();
	len2=str2.Len();
	

	pr=(DIRS *)Malloc(NumDIRS*(len1+1)*(len2+1)*sizeof(DIRS));

	score=CalculateMatrix_Diag(str1, str2, len1, len2, (int) (diag_fact*max(len1,len2))+1,(int) (diag_fact*max(len1,len2))+1, pr,entryVal);

	GetAlignments(pr, entryVal, str1, str2, al1, al2);

	free(pr);
}

void Needle::Align(char *str1,ProbString &str2,char *al1, ProbString &al2,float diag_fact)
{
	DIRS *pr;
	DIRS entryVal;
	float score;
	int len1,len2;
	int i,j;
	int *code1;
	
	len1=strlen(str1);
	len2=str2.Len();
	code1=StrToCode(str1);
	

	pr=(DIRS *)Malloc(NumDIRS*(len1+1)*(len2+1)*sizeof(DIRS));

	score=CalculateMatrix_Diag(code1, str2, len1, len2, (int) (diag_fact*max(len1,len2))+1,(int) (diag_fact*max(len1,len2))+1, pr,entryVal);

	GetAlignments(pr, entryVal, str1, str2, al1, al2);

	free(pr);
	free(code1);
}

void Needle::Align(ProbString &str1,ProbString &str2,ProbString &al1, ProbString &al2,int* pos1, int* pos2, float diag_fact)
{
	DIRS *pr;
	DIRS entryVal;
	float score;
	int len1,len2;
	int i,j;
	
	
	len1=str1.Len();
	len2=str2.Len();
	

	pr=(DIRS *)Malloc(NumDIRS*(len1+1)*(len2+1)*sizeof(DIRS));

	score=CalculateMatrix_Diag(str1, str2, len1, len2, (int) (diag_fact*max(len1,len2))+1,(int) (diag_fact*max(len1,len2))+1, pr,entryVal);
	GetAlignments(pr,entryVal, str1, str2, al1, al2, pos1, pos2);

	free(pr);
}

void Needle::AlignToTemp(char *str1,ProbString &str2,char *al1)
{
	DIRS *pr;
	DIRS entryVal;
	float score;
	int len1,len2;
	int i,j;
	int *code1;
	ProbString al2;
	
	len1=strlen(str1);
	len2=str2.Len();
	code1=StrToCode(str1);
	

	pr=(DIRS *)Malloc(NumDIRS*(len1+1)*(len2+1)*sizeof(DIRS));

	score=CalculateMatrix_NoDgap(code1, str2, len1, len2, pr, entryVal);
	
	GetAlignments(pr,entryVal, str1, str2, al1, al2);

	free(pr);
	free(code1);
}

