﻿using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;

public class GameGraphics
{
    private ArrayList _emptySquares = new ArrayList(40);
    private Bitmap _boardImage;
    private Bitmap[] _piece1 = new Bitmap[13], _piece2 = new Bitmap[13];
    private Board _board;
    private BoardSquare _lastClickedSquare, _clickedSquare, _attackedSquare;
    private bool _oneThroughNine = false;
    private Form _gameForm;
    private GameLogic _gameLogic;
    private int _pieceSelected=12;
    private int[] _setUpPieces = new int[12];
    private int[][] _piecesOnScreen = new int[10][];
    private LoadingScreen _loadingScreen;
    private MainMenu _mainMenu;
    private MenuItem _diplayOneThroughNineMenuItem, _drawProposalMenuItem, _finalizeMenuItem, _setUpMenu, _invertDisplayMenuItem, _showPiecesAtEndMenuItem;
    private Panel _boardPanel, _graveyardPanel;
    private Player _player;
    private Rectangle[] _graveyardSquares = new Rectangle[12];
    private Rectangle[][] _boardSquares = new Rectangle[10][];
    private SolidBrush _graveYardBrush1, _graveYardBrush2;
    private TextBox _chatBox, _chatField, _currentTurnField;
    private Timer _displayTimer;

    public GameGraphics(GameLogic gameLogic, bool chattingEnabled, bool graveyardEnabled, bool oneThroughNine, Board board, Player player, LoadingScreen loadingScreen)
    {
        _gameLogic = gameLogic;
        _oneThroughNine = oneThroughNine;
        _board = board;
        _player = player;
        _loadingScreen = loadingScreen;

        _graveYardBrush1 = new SolidBrush(_gameLogic.getPlayers()[0].getPlayerColor());
        _graveYardBrush2 = new SolidBrush(_gameLogic.getPlayers()[1].getPlayerColor());

        _boardImage = Stratego.Properties.Resources.strategoboard;
        _piece1[0] = Stratego.Properties.Resources.Bomb;
        _piece1[1] = Stratego.Properties.Resources.Spy;
        _piece1[2] = Stratego.Properties.Resources._2;
        _piece1[3] = Stratego.Properties.Resources._3;
        _piece1[4] = Stratego.Properties.Resources._4;
        _piece1[5] = Stratego.Properties.Resources._5;
        _piece1[6] = Stratego.Properties.Resources._6;
        _piece1[7] = Stratego.Properties.Resources._7;
        _piece1[8] = Stratego.Properties.Resources._8;
        _piece1[9] = Stratego.Properties.Resources._9;
        _piece1[10] = Stratego.Properties.Resources._10;
        _piece1[11] = Stratego.Properties.Resources.R_Flag;
        _piece1[12] = Stratego.Properties.Resources.B_Flag;
        _piece2[0] = Stratego.Properties.Resources.Bomb;
        _piece2[1] = Stratego.Properties.Resources.Spy;
        _piece2[2] = Stratego.Properties.Resources._2_Other;
        _piece2[3] = Stratego.Properties.Resources._3_Other;
        _piece2[4] = Stratego.Properties.Resources._4_Other;
        _piece2[5] = Stratego.Properties.Resources._5_Other;
        _piece2[6] = Stratego.Properties.Resources._6_Other;
        _piece2[7] = Stratego.Properties.Resources._7_Other;
        _piece2[8] = Stratego.Properties.Resources._8_Other;
        _piece2[9] = Stratego.Properties.Resources._9_Other;
        _piece2[10] = Stratego.Properties.Resources._10_Other;
        _piece2[11] = Stratego.Properties.Resources.R_Flag;
        _piece2[12] = Stratego.Properties.Resources.B_Flag;

        MenuItem gameOptionsMenu = new MenuItem("Game Options");

        _drawProposalMenuItem = new MenuItem("Propose Draw");
        _drawProposalMenuItem.Enabled = false;
        _drawProposalMenuItem.Click += new EventHandler(this.menuItemClicked);

        _showPiecesAtEndMenuItem = new MenuItem("Show Pieces at End");
        _showPiecesAtEndMenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem displayMenu = new MenuItem("Display");

        _diplayOneThroughNineMenuItem = new MenuItem("Display 1-9");
        if (_oneThroughNine == true)
        {
            _diplayOneThroughNineMenuItem.Text = "Display 2-10";
        }
        _diplayOneThroughNineMenuItem.Click += new EventHandler(this.menuItemClicked);

        _invertDisplayMenuItem = new MenuItem("Invert Display");
        _invertDisplayMenuItem.Click += new EventHandler(this.menuItemClicked);

        _setUpMenu = new MenuItem("Set-Up");

        _finalizeMenuItem = new MenuItem("Finalize Set-up");
        _finalizeMenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem randomMenuItem = new MenuItem("Random Set-up");
        randomMenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem presetMenuItem = new MenuItem("Preset Set-up");

        MenuItem preset1MenuItem = new MenuItem("Set-up #1");
        preset1MenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem preset2MenuItem = new MenuItem("Set-up #2");
        preset2MenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem preset3MenuItem = new MenuItem("Set-up #3");
        preset3MenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem preset4MenuItem = new MenuItem("Set-up #4");
        preset4MenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem preset5MenuItem = new MenuItem("Set-up #5");
        preset5MenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem mirrorMenuItem = new MenuItem("Mirror");
        mirrorMenuItem.Click += new EventHandler(this.menuItemClicked);

        MenuItem clearPiecesMenuItem = new MenuItem("Clear Pieces");
        clearPiecesMenuItem.Click += new EventHandler(this.menuItemClicked);

        gameOptionsMenu.MenuItems.AddRange(new MenuItem[] { _drawProposalMenuItem, _showPiecesAtEndMenuItem });

        displayMenu.MenuItems.AddRange(new MenuItem[] { _diplayOneThroughNineMenuItem, _invertDisplayMenuItem });

        presetMenuItem.MenuItems.AddRange(new MenuItem[] { preset1MenuItem, preset2MenuItem, preset3MenuItem, preset4MenuItem, preset5MenuItem });

        _setUpMenu.MenuItems.AddRange(new MenuItem[] { _finalizeMenuItem, randomMenuItem, presetMenuItem, mirrorMenuItem, clearPiecesMenuItem });

        _mainMenu = new MainMenu();
        _mainMenu.MenuItems.AddRange(new MenuItem[] { gameOptionsMenu, displayMenu, _setUpMenu });

        _gameForm = new Form();
        _gameForm.BackColor = Color.White;
        _gameForm.Text = "Stratego";
        _gameForm.Size = new Size(800, 620);
        _gameForm.StartPosition = FormStartPosition.CenterScreen;
        _gameForm.MaximizeBox = false;
        _gameForm.Menu = _mainMenu;
        _gameForm.KeyPreview = true;
        _gameForm.FormBorderStyle = FormBorderStyle.FixedSingle;
        _gameForm.Closing += new CancelEventHandler(this.closeEvent);

        _boardPanel = new Panel();
        _boardPanel.Size = new Size(600, 600);
        _boardPanel.BackColor = Color.Empty;
        _boardPanel.Location = new Point(200, 0);
        _boardPanel.Paint += new PaintEventHandler(this.paintBoard);
        _boardPanel.MouseClick += new MouseEventHandler(this.mouseEvent_MouseClick);

        _boardPanel.BackgroundImage = _boardImage;

        _graveyardPanel = new Panel();
        _graveyardPanel.Size = new Size(200, 350);
        _graveyardPanel.BackColor = Color.Black;
        _graveyardPanel.Location = new Point(0, 0);
        _graveyardPanel.Paint += new PaintEventHandler(this.paintGraveyard);
        _graveyardPanel.MouseClick += new MouseEventHandler(this.mouseEvent_MouseClick);
        _graveyardPanel.MouseDoubleClick +=new MouseEventHandler(this.mouseEvent_MouseDoubleClick);

        _currentTurnField = new TextBox();
        _currentTurnField.Size = new Size(200, 50);
        _currentTurnField.Location = new Point(0, 350);
        _currentTurnField.TextAlign = HorizontalAlignment.Center;
        _currentTurnField.Font = new Font("Times New Roman", 10, FontStyle.Bold);
        _currentTurnField.ReadOnly = true;
        _currentTurnField.BackColor = Color.Black;
        _currentTurnField.ForeColor = Color.Black;
        _currentTurnField.ScrollBars = ScrollBars.None;

        _chatBox = new TextBox();
        _chatBox.Size = new Size(200, 175);
        _chatBox.Location = new Point(0, 375);
        _chatBox.Font = new Font("Times New Roman", 10);
        _chatBox.ReadOnly = true;
        _chatBox.BackColor = Color.Black;
        _chatBox.ForeColor = Color.White;
        _chatBox.Multiline = true;
        _chatBox.ScrollBars = ScrollBars.Vertical;

        _chatField = new TextBox();
        _chatField.Size = new Size(200, 50);
        _chatField.Font = new Font("Times New Roman", 10);
        _chatField.Location = new Point(0, 550);
        _chatField.BackColor = Color.Black;
        _chatField.ForeColor = Color.White;
        _chatField.KeyDown += new KeyEventHandler(this.chatFieldTextEntered);

        //if the option for chatting has been disabled
        if (chattingEnabled == false)
        {
            _chatBox.Enabled = false;
            _chatField.Enabled = false;
            _chatField.BackColor = Color.Gray;
            _chatBox.BackColor = Color.Gray;
        }

        _displayTimer = new Timer();
        _displayTimer.Interval = 2000;
        _displayTimer.Tick += new EventHandler(this.displayPiece);

        //if the player is the blue player
        if (_player.getPlayerColor().Equals(Color.Blue))
        {
            int indexX = 0;
            for (int x = _boardSquares.Length - 1; x >= 0; x--)
            {
                _boardSquares[x] = new Rectangle[10];
                _piecesOnScreen[x] = new int[10];
                int indexY = 0;
                for (int y = _boardSquares[x].Length - 1; y >= 0; y--)
                {
                    _boardSquares[x][y] = new Rectangle();
                    _boardSquares[x][y].Height = 57;
                    _boardSquares[x][y].Width = 59;
                    _boardSquares[x][y].Location = new Point((1 + (indexY * _boardSquares[x][y].Width)), (2 + (indexX * _boardSquares[x][y].Height)));
                    _piecesOnScreen[x][y] = 0;
                    indexY++;
                }
                indexX++;
            }

            for (int x = 0; x < 4; x++)
            {
                for (int y = 0; y < _boardSquares.Length; y++)
                {
                    _emptySquares.Add(_board.getBoardSquares()[x][y]);
                }
            }
        }
        //if the player is the red player
        else
        {
            for (int x = 0; x < _boardSquares.Length; x++)
            {
                _boardSquares[x] = new Rectangle[10];
                _piecesOnScreen[x] = new int[10];
                for (int y = 0; y < _boardSquares.Length; y++)
                {
                    _boardSquares[x][y] = new Rectangle();
                    _boardSquares[x][y].Height = 57;
                    _boardSquares[x][y].Width = 59;
                    _boardSquares[x][y].Location = new Point((1 + (y * _boardSquares[x][y].Width)), (2 + (x * _boardSquares[x][y].Height)));
                    _piecesOnScreen[x][y] = 0;
                }
            }

            for (int x = 6; x < _boardSquares.Length; x++)
            {
                for (int y = 0; y < _boardSquares.Length; y++)
                {
                    _emptySquares.Add(_board.getBoardSquares()[x][y]);
                }
            }
        }

        for (int x = 0; x < _graveyardSquares.Length; x++)
        {
            _graveyardSquares[x] = new Rectangle();
            _graveyardSquares[x].Height = 58;
            _graveyardSquares[x].Width = 99;
            if (x % 2 == 0)
            {
                _graveyardSquares[x].Location = new Point(0, ((x / 2) * _graveyardSquares[x].Height));
            }
            else
            {
                _graveyardSquares[x].Location = new Point(_graveyardSquares[x].Width, ((x / 2) * _graveyardSquares[x].Height));
            }
        }

        _setUpPieces[0] = 6;
        _setUpPieces[1] = 1;
        _setUpPieces[2] = 8;
        _setUpPieces[3] = 5;
        _setUpPieces[4] = 4;
        _setUpPieces[5] = 4;
        _setUpPieces[6] = 4;
        _setUpPieces[7] = 3;
        _setUpPieces[8] = 2;
        _setUpPieces[9] = 1;
        _setUpPieces[10] = 1;
        _setUpPieces[11] = 1;

        _gameForm.Controls.Add(_graveyardPanel);
        _gameForm.Controls.Add(_currentTurnField);
        _gameForm.Controls.Add(_chatBox);
        _gameForm.Controls.Add(_chatField);
        _gameForm.Controls.Add(_boardPanel);
        _gameForm.Show();

        _chatField.Focus();
    }

//**** delete after making AI or making mulitplayer
    public void setPlayer(Player player)
    {
        _player = player;
    }

    public void startGame()
    {
        _graveyardPanel.MouseClick -= new MouseEventHandler(this.mouseEvent_MouseClick);
        _graveyardPanel.MouseDoubleClick -= new MouseEventHandler(this.mouseEvent_MouseDoubleClick);
        _emptySquares = null;
        _lastClickedSquare = null;
        _pieceSelected = 12;
        _mainMenu.MenuItems.Remove(_setUpMenu);
        _drawProposalMenuItem.Enabled = true;

        //if the game is single player
        if (_gameLogic.isSoloGame() == true)
        {
            //if you are the blue player
            if (_player.getPlayerColor().Equals(Color.Blue))
            {
                //makes it so the enemy pieces are Xs
                for (int x = 6; x < 10; x++)
                {
                    for (int y = 0; y < _piecesOnScreen[x].Length; y++)
                    {
                        _piecesOnScreen[x][y] = 2;
                    }
                }

                //makes it so that your own pieces are drawn
                for (int x = 0; x < 4; x++)
                {
                    for (int y = 0; y < _piecesOnScreen[x].Length; y++)
                    {
                        _piecesOnScreen[x][y] = 1;
                    }
                }
            }
            //if you are the red player
            else
            {
                //makes it so the enemy pieces are Xs
                for (int x = 0; x < 4; x++)
                {
                    for (int y = 0; y < _piecesOnScreen[x].Length; y++)
                    {
                        _piecesOnScreen[x][y] = 2;
                    }
                }

                //makes it so that your own pieces are drawn
                for (int x = 6; x < 10; x++)
                {
                    for (int y = 0; y < _piecesOnScreen[x].Length; y++)
                    {
                        _piecesOnScreen[x][y] = 1;
                    }
                }
            }
        }
        _currentTurnField.BackColor = Color.Red;
        _currentTurnField.Text = _gameLogic.getCurrentPlayer().getPlayerName() + "'s turn!";
        _boardPanel.Invalidate();
        _graveyardPanel.Invalidate();
    }

    public void appendToChatBox(String message)
    {
        _chatBox.AppendText(message + "\n");
    }

    public int[][] getPiecesOnScreen()
    {
        return _piecesOnScreen;
    }

    public void setCurrentTurnField(Player currentPlayer)
    {
        _currentTurnField.BackColor = currentPlayer.getPlayerColor();
        _currentTurnField.Text = currentPlayer.getPlayerName() + "'s turn!";
    }

    public void setCurrentTurnFieldEndGame(String message, Color winningPlayerColor)
    {
        _currentTurnField.BackColor = winningPlayerColor;
        _currentTurnField.Text = message;
    }

    public delegate void setPlayerFinalizedAlertDelegate();

    public void setPlayerFinalizedAlert()
    {
        _currentTurnField.BackColor = Color.White;

        //if this player is the red player
        if (_player.getPlayerColor().Equals(Color.Red))
        {
            _currentTurnField.Text = "Blue has finalized thier set-up!";
        }
        //if this player is the blue player
        else
        {
            _currentTurnField.Text = "Red has finalized thier set-up!";
        }
    }

    public void invalidateBoardPanel()
    {
        _boardPanel.Invalidate();
    }

    public Form getGameForm()
    {
        return _gameForm;
    }

    public delegate void displayEndMessageDelegate(String message);

    public void displayEndMessage(String message)
    {
        DialogResult result = MessageBox.Show(_gameForm, message + "\n" + "Exit game then go to Loading Screen?", "Game Over", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Exclamation);

        //exit game or go to loading screen
        if (result == DialogResult.Yes || result == DialogResult.No)
        {
            //if the game is single player
            if (_gameLogic.isSoloGame() == true)
            {
                _displayTimer.Stop();
                _gameForm.Dispose();
                _lastClickedSquare = null;
                _attackedSquare = null;
                _boardSquares = null;
                _board = null;
                _player = null;
                _gameLogic = null;

                //go to loading screen
                if (result == DialogResult.Yes)
                {
                    _loadingScreen.getLoadingForm().Show();
                    _loadingScreen.createLoadingScreen();
                    GC.Collect(GC.MaxGeneration, GCCollectionMode.Forced);
                }
                //exit game
                else
                {
                    _loadingScreen.getBindingSocket().Close();
                    System.Environment.Exit(0);
                }
            }
            //if the game is multiplayer
            else
            {
                _displayTimer.Stop();
                if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                {
                    _gameLogic.getStrategoClient().leaveStrategoGame();
                }
                _gameForm.Dispose();
                _lastClickedSquare = null;
                _attackedSquare = null;
                _boardSquares = null;
                _board = null;
                _player = null;
                _gameLogic = null;

                //go to loading screen
                if (result == DialogResult.Yes)
                {
                    _loadingScreen.getLoadingForm().Show();
                    _loadingScreen.createLoadingScreen();
                    GC.Collect(GC.MaxGeneration, GCCollectionMode.Forced);
                }
                //exit game
                else
                {
                    _loadingScreen.getBindingSocket().Close();
                    System.Environment.Exit(0);
                }
            }
        }
    }

    public void setLastClickedSquare(int index)
    {
        if (index != 200)
        {
            _lastClickedSquare = _board.getBoardSquares()[index / 10][index % 10];
        }
        else
        {
            _lastClickedSquare = null;
        }
        _boardPanel.Invalidate();
    }

    public void setAttackedSquare(int index)
    {
        if (index != 200)
        {
            _attackedSquare = _board.getBoardSquares()[index / 10][index % 10];
        }
    }

    private void closeEvent(object sender, CancelEventArgs cArgs)
    {
        DialogResult result = MessageBox.Show(_gameForm, "Exit game then go to Loading Screen?", "Exiting Game", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
        
        //exit game or go to loading screen
        if (result == DialogResult.Yes || result == DialogResult.No)
        {
            //if the game is single player
            if (_gameLogic.isSoloGame() == true)
            {
                _displayTimer.Stop();
                _gameForm.Dispose();
                _lastClickedSquare = null;
                _attackedSquare = null;
                _boardSquares = null;
                _board = null;
                _player = null;
                _gameLogic = null;

                //go to loading screen
                if (result == DialogResult.Yes)
                {
                    _loadingScreen.getLoadingForm().Show();
                    _loadingScreen.createLoadingScreen();
                    GC.Collect(GC.MaxGeneration, GCCollectionMode.Forced);
                }
                //exit game
                else
                {
                    _loadingScreen.getBindingSocket().Close();
                    System.Environment.Exit(0);
                }
            }
            //if the game is multiplayer
            else
            {
                _displayTimer.Stop();
                if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                {
                    _gameLogic.getStrategoClient().leaveStrategoGame();
                }
                _gameForm.Dispose();
                _lastClickedSquare = null;
                _attackedSquare = null;
                _boardSquares = null;
                _board = null;
                _player = null;
                _gameLogic = null;

                //go to loading screen
                if (result == DialogResult.Yes)
                {
                    _loadingScreen.getLoadingForm().Show();
                    _loadingScreen.createLoadingScreen();
                    GC.Collect(GC.MaxGeneration, GCCollectionMode.Forced);
                }
                //exit game
                else
                {
                    _loadingScreen.getBindingSocket().Close();
                    System.Environment.Exit(0);
                }
            }
        }
        //cancel the exiting of the game
        else
        {
            cArgs.Cancel = true;
        }
    }

    public void paintBoard(object sender, PaintEventArgs e)
    {
        Graphics g = e.Graphics;

        //if the game is in the set-up phase
        if (_gameLogic.getPhase() == 0)
        {
            //paint the last clicked square black
            if (_lastClickedSquare != null)
            {
                g.FillRectangle(Brushes.Black, _boardSquares[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10]);
            }

            for (int x = 0; x < _boardSquares.Length; x++)
            {
                for (int y = 0; y < _boardSquares[x].Length; y++)
                {
                    //if there is a piece in the square
                    if (_board.getBoardSquares()[x][y].getPiece() != null)
                    {
                        //if the pieces are 2-10
                        if (_oneThroughNine == false)
                        {
                            //if the player is blue, paint the blue player's flag
                            if (_board.getBoardSquares()[x][y].getPiece().getRank() == 11 && _player.getPlayerColor().Equals(Color.Blue))
                            {
                                g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank() + 1], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                            //paint the rest of the pieces normally
                            else
                            {
                                g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                        }
                        //if the pieces are 1-9
                        else
                        {
                            //if the player is blue, paint the blue player's flag
                            if (_board.getBoardSquares()[x][y].getPiece().getRank() == 11 && _player.getPlayerColor().Equals(Color.Blue))
                            {
                                g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank() + 1], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                            //paint the rest of the pieces normally
                            else
                            {
                                g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                        }
                    }
                }
            }
        }
        else if (_gameLogic.getPhase() == 1)
        {
            //the last clicked square
            if (_lastClickedSquare != null)
            {
                g.FillRectangle(Brushes.Black, _boardSquares[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10]);
            }

            //the square being attacked
            if (_attackedSquare != null)
            {
                g.FillRectangle(Brushes.Black, _boardSquares[_attackedSquare.getIndex() / 10][_attackedSquare.getIndex() % 10]);
            }

            for (int x = 0; x < _piecesOnScreen.Length; x++)
            {
                for (int y = 0; y < _piecesOnScreen[x].Length; y++)
                {
                    //if they are your pieces
                    if (_piecesOnScreen[x][y] == 1)
                    {
                        //if you are the red player
                        if (_player.getPlayerColor().Equals(Color.Red))
                        {
                            /*SolidBrush redBrush = new SolidBrush(Color.Red);
                            g.FillRectangle(redBrush, _boardSquares[x][y]);
                            Pen blackPen = new Pen(Color.Black);
                            g.DrawRectangle(blackPen, _boardSquares[x][y]);*/
                            //if the pieces are 2-10
                            if (_oneThroughNine == false)
                            {
                                g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                            //if the pieces are 1-9
                            else
                            {
                                g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                        }
                        //if you are the blue player
                        else
                        {
                            //if the pieces are 2-10
                            if (_oneThroughNine == false)
                            {
                                //if the piece is not null
                                if (_board.getBoardSquares()[x][y].getPiece() != null)
                                {
                                    //if it is the blue flag
                                    if (_board.getBoardSquares()[x][y].getPiece().getRank() == 11)
                                    {

                                        g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank() + 1], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                    //if the piece is a normal piece
                                    else
                                    {
                                        g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                }
                            }
                            //if the pieces are 1-9
                            else
                            {
                                //if the piece is not null
                                if (_board.getBoardSquares()[x][y].getPiece() != null)
                                {
                                    //if it is the blue flag
                                    if (_board.getBoardSquares()[x][y].getPiece().getRank() == 11)
                                    {
                                        g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank() + 1], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                    //if the piece is a normal piece
                                    else
                                    {
                                        g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                }
                            }    
                        }
                    }
                    //if they are the enemies pieces
                    else if (_piecesOnScreen[x][y] == 2)
                    {
                        /*SolidBrush redBrush = new SolidBrush(Color.Blue);
                        g.FillRectangle(redBrush, _boardSquares[x][y]);
                        Pen blackPen = new Pen(Color.Black);
                        g.DrawRectangle(blackPen, _boardSquares[x][y]);*/
                                                //if you are the red player
                        if (_player.getPlayerColor().Equals(Color.Red))
                        {
                            g.DrawString("X", new Font("Arial", 34), Brushes.Blue, new Point(_boardSquares[x][y].X + 6, _boardSquares[x][y].Y + 6));
                        }
                        else
                        {
                            g.DrawString("X", new Font("Arial", 34), Brushes.Red, new Point(_boardSquares[x][y].X + 6, _boardSquares[x][y].Y + 6));
                        }
                    }
                    //if the enemies piece was attacked
                    else if (_piecesOnScreen[x][y] == 3)
                    {
                        //if you are the red player
                        if (_player.getPlayerColor().Equals(Color.Red))
                        {
                            //if the pieces are 2-10
                            if (_oneThroughNine == false)
                            {
                                //if the piece is not null
                                if (_board.getBoardSquares()[x][y].getPiece() != null)
                                {
                                    //if it is the blue flag
                                    if (_board.getBoardSquares()[x][y].getPiece().getRank() == 11)
                                    {

                                        g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank() + 1], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                    //if the piece is a normal piece
                                    else
                                    {
                                        g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                }
                            }
                            //if the pieces are 1-9
                            else
                            {
                                //if the piece is not null
                                if (_board.getBoardSquares()[x][y].getPiece() != null)
                                {
                                    //if it is the blue flag
                                    if (_board.getBoardSquares()[x][y].getPiece().getRank() == 11)
                                    {
                                        g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank() + 1], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                    //if the piece is a normal piece
                                    else
                                    {
                                        g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                                    }
                                }
                            }
                        }
                        //if you are the blue player
                        else
                        {
                            //if the pieces are 2-10
                            if (_oneThroughNine == false)
                            {
                                g.DrawImage(_piece1[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                            //if the pieces are 1-9
                            else
                            {
                                g.DrawImage(_piece2[_board.getBoardSquares()[x][y].getPiece().getRank()], _boardSquares[x][y].X + 10, _boardSquares[x][y].Y + 4, 40, 50);
                            }
                        }
                    }
                }
            }
        }
        g.Dispose();
    }

    private void paintGraveyard(object sender, PaintEventArgs e)
    {
        Graphics g = e.Graphics;
        //if the game is in the set-up phase
        if (_gameLogic.getPhase() == 0)
        {
            for (int x = 0; x < _graveyardSquares.Length; x++)
            {
                //if the player is the blue player, draw his flag
                if (x == 11 && _player.getPlayerColor().Equals(Color.Blue))
                {
                    g.DrawImage(_piece1[x + 1], _graveyardSquares[x].X, _graveyardSquares[x].Y, 50, 58);
                }
                //draws the piece images normally
                else
                {
                    //if the player picked the 1-9 view
                    if (_oneThroughNine == false)
                    {
                        g.DrawImage(_piece1[x], _graveyardSquares[x].X, _graveyardSquares[x].Y, 50, 58);
                    }
                    //if the player picked the 2-10 view
                    else
                    {
                        g.DrawImage(_piece2[x], _graveyardSquares[x].X, _graveyardSquares[x].Y, 50, 58);
                    }
                }
                //if you clicked on a piece
                if (_pieceSelected < 12 && _pieceSelected == x)
                {
                    g.DrawString(_setUpPieces[x].ToString(), new Font("Arial", 15), Brushes.White, new Point(_graveyardSquares[x].X + 60, _graveyardSquares[x].Y));
                }
                //if you didn't click on a piece
                else
                {
                    //if the player is player 1
                    if (_player.getPlayerColor().Equals(Color.Red))
                    {
                        g.DrawString(_setUpPieces[x].ToString(), new Font("Arial", 15), Brushes.Red, new Point(_graveyardSquares[x].X + 60, _graveyardSquares[x].Y));
                    }
                    //if the player is player 2
                    else
                    {
                        g.DrawString(_setUpPieces[x].ToString(), new Font("Arial", 15), Brushes.Blue, new Point(_graveyardSquares[x].X + 60, _graveyardSquares[x].Y));
                    }
                }
                g.DrawRectangle(new Pen(Color.White, 1), _graveyardSquares[x]);
            }
        }
        else if (_gameLogic.getPhase() == 1)
        {
            for (int x = 0; x < _graveyardSquares.Length; x++)
            {
                //if the player is the blue player, draw his flag
                if (x == 11 && _player.getPlayerColor().Equals(Color.Blue))
                {
                    g.DrawImage(_piece1[x + 1], _graveyardSquares[x].X, _graveyardSquares[x].Y, 50, 58);
                }
                //draws the piece images normally
                else
                {
                    //if the player picked the 1-9 view
                    if (_oneThroughNine == false)
                    {
                        g.DrawImage(_piece1[x], _graveyardSquares[x].X, _graveyardSquares[x].Y, 50, 58);
                    }
                    //if the player picked the 2-10 view
                    else
                    {
                        g.DrawImage(_piece2[x], _graveyardSquares[x].X, _graveyardSquares[x].Y, 50, 58);
                    }
                }

                g.DrawString(_gameLogic.getPlayers()[0].getPiecesInGraveard()[x].ToString(), new Font("Arial", 15), _graveYardBrush1, new Point(_graveyardSquares[x].X + 50, _graveyardSquares[x].Y));
                g.DrawString(_gameLogic.getPlayers()[1].getPiecesInGraveard()[x].ToString(), new Font("Arial", 15), _graveYardBrush2, new Point(_graveyardSquares[x].X + 75, _graveyardSquares[x].Y));
                
                g.DrawRectangle(new Pen(Color.White, 1), _graveyardSquares[x]);
            }
        }
        g.Dispose();
    }

    //when the player presses a key, this checks to see if it was the enter key
    private void chatFieldTextEntered(object sender, KeyEventArgs e)
    {
        //if enter is pressed and the player has entered a message
        if (e.KeyCode == Keys.Enter && _chatField.Text.CompareTo("") != 0)
        {
            if (_gameLogic.isSoloGame() == true)
            {
                _chatBox.AppendText(_player.getPlayerName() + ": " + _chatField.Text + "\n");
            }
            else
            {
                if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                {
                    _gameLogic.getStrategoClient().writeToStrategoGameChat(_player.getPlayerName() + ": " + _chatField.Text);
                }
            }
            _chatField.Text = "";
            e.SuppressKeyPress = true;
        }
    }

    //deals with when the mouse button is clicked
    private void mouseEvent_MouseClick(object sender, MouseEventArgs e)
    {
        //if the game is in the set-up phase
        if (_gameLogic.getPhase() == 0 && _setUpMenu.Enabled == true)
        {
            //if the left mouse button is clicked
            if (e.Button.Equals(MouseButtons.Left))
            {
                if (sender.Equals(_boardPanel))
                {
                    BoardSquare clickedSquare = null;
                    for (int x = 0; x < _boardSquares.Length; x++)
                    {
                        for (int y = 0; y < _boardSquares[x].Length; y++)
                        {
                            if (_boardSquares[x][y].Contains(e.Location))
                            {
                                clickedSquare = _board.getBoardSquares()[x][y];
                            }
                        }
                    }
                    //if the clicked square is one of your starting squares
                    if (clickedSquare != null && ((clickedSquare.getIndex() > 59 && _player.getPlayerColor().Equals(Color.Red)) || (clickedSquare.getIndex() < 40 && _player.getPlayerColor().Equals(Color.Blue))))
                    {
                        //if the square doesn't have a piece and you want to put a piece there
                        if (clickedSquare.getPiece() == null && _pieceSelected < 12 && _setUpPieces[_pieceSelected] > 0)
                        {
                            //if you aren't trying to swap pieces
                            if (_lastClickedSquare == null)
                            {
                                clickedSquare.setPiece(new Piece(_pieceSelected, _player, clickedSquare));
                                _setUpPieces[_pieceSelected]--;
                                if (_setUpPieces[_pieceSelected] == 0)
                                {
                                    _pieceSelected = 12;
                                }
                            }
                            //if you are trying to swap pieces
                            else
                            {
                                _emptySquares.Add(_lastClickedSquare);
                                clickedSquare.setPiece(_lastClickedSquare.getPiece());
                                clickedSquare.getPiece().setCurrentSquare(clickedSquare);
                                _lastClickedSquare.setPiece(null);
                                _setUpPieces[_pieceSelected]--;
                                _pieceSelected = 12;
                                _lastClickedSquare = null;
                            }
                            _emptySquares.Remove(clickedSquare);
                            _graveyardPanel.Invalidate();
                            _boardPanel.Invalidate();
                        }
                        //if the square already has a piece and you want to put a piece there
                        else if (clickedSquare.getPiece() != null && _pieceSelected < 12 && _setUpPieces[_pieceSelected] > 0)
                        {
                            //if you aren't trying to swap pieces
                            if (_lastClickedSquare == null)
                            {
                                Piece piece = clickedSquare.getPiece();
                                clickedSquare.setPiece(new Piece(_pieceSelected, _player, clickedSquare));
                                _setUpPieces[_pieceSelected]--;
                                _pieceSelected = piece.getRank();
                                _setUpPieces[_pieceSelected]++;
                            }
                            //if you are trying to swap pieces
                            else
                            {
                                Piece piece = clickedSquare.getPiece();
                                clickedSquare.setPiece(_lastClickedSquare.getPiece());
                                clickedSquare.getPiece().setCurrentSquare(clickedSquare);
                                _lastClickedSquare.setPiece(piece);
                                _lastClickedSquare.getPiece().setCurrentSquare(_lastClickedSquare);
                                _setUpPieces[_pieceSelected]--;
                                _pieceSelected = 12;
                                _lastClickedSquare = null;
                            }
                            _graveyardPanel.Invalidate();
                            _boardPanel.Invalidate();
                        }
                        //if the square already has a piece and you select it
                        else if (clickedSquare.getPiece() != null && _pieceSelected == 12)
                        {
                            _lastClickedSquare = clickedSquare;
                            _pieceSelected = clickedSquare.getPiece().getRank();
                            _setUpPieces[_pieceSelected]++;
                            _graveyardPanel.Invalidate();
                            _boardPanel.Invalidate();
                        }
                    }
                    else
                    {
                        //if you are trying to perform a swap
                        if (_lastClickedSquare != null)
                        {
                            _setUpPieces[_pieceSelected]--;
                            _lastClickedSquare = null;
                            _graveyardPanel.Invalidate();
                            _boardPanel.Invalidate();
                        }
                        //if you are trying to place a unit
                        if (_pieceSelected < 12)
                        {
                            _pieceSelected = 12;
                            _graveyardPanel.Invalidate();
                            _boardPanel.Invalidate();
                        }
                    }
                }
                else if (sender.Equals(_graveyardPanel))
                {
                    for (int x = 0; x < _graveyardSquares.Length; x++)
                    {
                        if (_graveyardSquares[x].Contains(e.Location))
                        {
                            if (_setUpPieces[x] > 0)
                            {
                                _pieceSelected = x;
                                _graveyardPanel.Invalidate();
                            }
                        }
                    }
                }
            }
            //if the right mouse button is clicked
            else if (e.Button.Equals(MouseButtons.Right))
            {
                if (sender.Equals(_boardPanel))
                {
                    BoardSquare clickedSquare = null;
                    for (int x = 0; x < _boardSquares.Length; x++)
                    {
                        for (int y = 0; y < _boardSquares[x].Length; y++)
                        {
                            if (_boardSquares[x][y].Contains(e.Location))
                            {
                                clickedSquare = _board.getBoardSquares()[x][y];
                            }
                        }
                    }
                    //if you are clicking in your set-up area
                    if (clickedSquare != null && ((clickedSquare.getIndex() > 59 && _player.getPlayerColor().Equals(Color.Red)) || (clickedSquare.getIndex() < 40 && _player.getPlayerColor().Equals(Color.Blue))))
                    {
                        //if there is a piece
                        if (clickedSquare.getPiece() != null)
                        {
                            //if you are not trying to swap pieces
                            if (_lastClickedSquare == null)
                            {
                                _setUpPieces[clickedSquare.getPiece().getRank()]++;
                            }
                            //if you where trying to swap peices
                            else
                            {
                                //if the two pieces where different
                                if (_pieceSelected != clickedSquare.getPiece().getRank())
                                {
                                    _setUpPieces[_pieceSelected]--;
                                    _pieceSelected = clickedSquare.getPiece().getRank();
                                    _setUpPieces[_pieceSelected]++;
                                }
                            }
                            clickedSquare.setPiece(null);
                            _lastClickedSquare = null;
                            _emptySquares.Add(clickedSquare);
                            _graveyardPanel.Invalidate();
                            _boardPanel.Invalidate();
                        }
                        //if there is no piece
                        else
                        {
                            if (_lastClickedSquare != null)
                            {
                                _lastClickedSquare = null;
                                _pieceSelected = 12;
                                _graveyardPanel.Invalidate();
                                _boardPanel.Invalidate();
                            }
                        }
                    }
                    else
                    {
                        if (_lastClickedSquare != null)
                        {
                            _lastClickedSquare = null;
                            if (_pieceSelected < 12)
                            {
                                _setUpPieces[_pieceSelected]--;
                            }
                            _pieceSelected = 12;
                            _graveyardPanel.Invalidate();
                            _boardPanel.Invalidate();
                        }
                    }
                }
            }
        }
        //if the game is in the order phase and it is this player's turn
        else if (_gameLogic.getPhase() == 1 && _gameLogic.isYourTurn() == true && _gameLogic.hasGameEnded() == false)
        {
            if (e.Button.Equals(MouseButtons.Left))
            {
                BoardSquare clickedSquare = null;
                for (int x = 0; x < _boardSquares.Length; x++)
                {
                    for (int y = 0; y < _boardSquares[x].Length; y++)
                    {
                        if (_boardSquares[x][y].Contains(e.Location))
                        {
                            clickedSquare = _board.getBoardSquares()[x][y];
                        }
                    }
                }
                //if the square is somehow not equal to null
                if (clickedSquare != null)
                {
                    //if you did not already click a square
                    if (_lastClickedSquare == null)
                    {
                        //if the clicked square has a piece that belongs to this player
                        if (clickedSquare.getPiece() != null && clickedSquare.getPiece().getPlayer().Equals(_player))
                        {
                            //send to server the piece clicked
                            if (_gameLogic.isSoloGame() == false)
                            {
                                if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                                {
                                    _gameLogic.getStrategoClient().writePieceClicked(clickedSquare.getIndex());
                                }
                            }

                            _lastClickedSquare = clickedSquare;
                            _boardPanel.Invalidate();
                        }
                    }
                    //if you have already clicked a square
                    else
                    {
                        //if the clicked square has a piece
                        if (clickedSquare.getPiece() != null)
                        {
                            //if it belongs to this player
                            if (clickedSquare.getPiece().getPlayer().Equals(_player))
                            {
                                //send to server the piece clicked
                                if (_gameLogic.isSoloGame() == false)
                                {
                                    if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                                    {
                                        _gameLogic.getStrategoClient().writePieceClicked(clickedSquare.getIndex());
                                    }
                                }

                                _lastClickedSquare = clickedSquare;
                                _boardPanel.Invalidate();
                            }
                            //if it belongs to the other player
                            else
                            {
                                //if the piece is not a bomb or a flag
                                if (_lastClickedSquare.getPiece() != null && _lastClickedSquare.getPiece().getRank() > 0 && _lastClickedSquare.getPiece().getRank() < 11)
                                {
                                    //if the game is single player
                                    if (_gameLogic.isSoloGame() == true)
                                    {
                                        //if the clicked square is a legal move for this piece
                                        if (_lastClickedSquare.getOneMoveSquares().Contains(clickedSquare))
                                        {
                                            _piecesOnScreen[clickedSquare.getIndex() / 10][clickedSquare.getIndex() % 10] = 3;
                                            _clickedSquare = clickedSquare;
                                            _attackedSquare = clickedSquare;
                                            _displayTimer.Start();
                                            _boardPanel.Invalidate();
                                            _graveyardPanel.Invalidate();
                                        }
                                        //if it is a scout and it might be trying to move multiple spaces
                                        else if (_lastClickedSquare.getPiece().getRank() == 2)
                                        {
                                            //check and see if scout move is valid
                                            if (_gameLogic.validScoutMove(_lastClickedSquare, clickedSquare) == true)
                                            {
                                                _piecesOnScreen[clickedSquare.getIndex() / 10][clickedSquare.getIndex() % 10] = 3;
                                                _clickedSquare = clickedSquare;
                                                _attackedSquare = clickedSquare;
                                                _displayTimer.Start();
                                                _boardPanel.Invalidate();
                                                _graveyardPanel.Invalidate();
                                            }
                                        }
                                    }
                                    //if the game is multiplayer
                                    else
                                    {
                                        if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                                        {
                                            _gameLogic.getStrategoClient().writeMoveToServer(true, _lastClickedSquare.getIndex(), clickedSquare.getIndex());
                                        }
                                    }
                                }
                            }
                        }
                        //if the clicked square does not have a piece
                        else
                        {
                            //if the piece is not a bomb or a flag
                            if (_lastClickedSquare.getPiece() != null && _lastClickedSquare.getPiece().getRank() > 0 && _lastClickedSquare.getPiece().getRank() < 11)
                            {
                                //if the game is single player
                                if (_gameLogic.isSoloGame() == true)
                                {
                                    //if the clicked square is a legal move for this piece
                                    if (_lastClickedSquare.getOneMoveSquares().Contains(clickedSquare))
                                    {
                                        clickedSquare.setPiece(_lastClickedSquare.getPiece());
                                        clickedSquare.getPiece().setCurrentSquare(clickedSquare);
                                        if (_player.Equals(_gameLogic.getPlayers()[0]))
                                        {
                                            _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                                            _piecesOnScreen[clickedSquare.getIndex() / 10][clickedSquare.getIndex() % 10] = 1;
                                        }
                                        else
                                        {
                                            _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                                            _piecesOnScreen[clickedSquare.getIndex() / 10][clickedSquare.getIndex() % 10] = 2;
                                        }
                                        _lastClickedSquare.setPiece(null);
                                        _lastClickedSquare = null;
                                        _boardPanel.Invalidate();
                                        _gameLogic.nextTurn();
                                    }
                                    //if it is a scout and it might be trying to move multiple spaces
                                    else if (_lastClickedSquare.getPiece().getRank() == 2)
                                    {
                                        //check and see if scout move is valid
                                        if (_gameLogic.validScoutMove(_lastClickedSquare, clickedSquare) == true)
                                        {
                                            clickedSquare.setPiece(_lastClickedSquare.getPiece());
                                            clickedSquare.getPiece().setCurrentSquare(clickedSquare);
                                            if (_player.Equals(_gameLogic.getPlayers()[0]))
                                            {
                                                _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                                                _piecesOnScreen[clickedSquare.getIndex() / 10][clickedSquare.getIndex() % 10] = 1;
                                            }
                                            else
                                            {
                                                _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                                                _piecesOnScreen[clickedSquare.getIndex() / 10][clickedSquare.getIndex() % 10] = 2;
                                            }
                                            _lastClickedSquare.setPiece(null);
                                            _lastClickedSquare = null;
                                            _boardPanel.Invalidate();
                                            _gameLogic.nextTurn();
                                        }
                                    }
                                }
                                //if the game is multiplayer
                                else
                                {
                                    if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                                    {
                                        _gameLogic.getStrategoClient().writeMoveToServer(false, _lastClickedSquare.getIndex(), clickedSquare.getIndex());
                                    }
                                }
                            }
                        }
                    }
                }
            }
            //if the right mouse button is clicked
            else if (e.Button.Equals(MouseButtons.Right))
            {
                //send to server the piece clicked
                if (_gameLogic.isSoloGame() == false)
                {
                    if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                    {
                        _gameLogic.getStrategoClient().writePieceClicked(200);
                    }
                }

                _lastClickedSquare = null;
                _boardPanel.Invalidate();
            }
        }
    }

    //deals with when the mouse button is double clicked
    private void mouseEvent_MouseDoubleClick(object sender, MouseEventArgs e)
    {
        //if the game is in the set-up phase
        if (_gameLogic.getPhase() == 0 && _setUpMenu.Enabled == true)
        {
            //if the left mouse button is clicked
            if (e.Button.Equals(MouseButtons.Left))
            {
                if (sender.Equals(_graveyardPanel))
                {
                    for (int x = 0; x < _graveyardSquares.Length; x++)
                    {
                        if (_graveyardSquares[x].Contains(e.Location))
                        {
                            //if there are more than 0 of the selected piece and you are not trying to perform a unit swap
                            if (_setUpPieces[x] > 0 && _lastClickedSquare == null)
                            {
                                int index = new Random().Next(_emptySquares.Count);
                                if (((BoardSquare)_emptySquares[index]).getPiece() == null)
                                {
                                    ((BoardSquare)_emptySquares[index]).setPiece(new Piece(x, _player, ((BoardSquare)_emptySquares[index])));
                                    _emptySquares.RemoveAt(index);
                                    _setUpPieces[x]--;
                                    _pieceSelected = 12;
                                    _graveyardPanel.Invalidate();
                                    _boardPanel.Invalidate();
                                }
                            }
                        }
                    }
                }
            }
        }
        //if the game is in the order phase
        else if (_gameLogic.getPhase() == 1 && _gameLogic.isYourTurn() == true && _gameLogic.hasGameEnded() == false)
        {
            //if it is this player's turn
            if (_gameLogic.isYourTurn() == true)
            {
                BoardSquare clickedSquare = null;
                for (int x = 0; x < _boardSquares.Length; x++)
                {
                    for (int y = 0; y < _boardSquares[x].Length; y++)
                    {
                        if (_boardSquares[x][y].Contains(e.Location))
                        {
                            clickedSquare = _board.getBoardSquares()[x][y];
                        }
                    }
                }

                if (clickedSquare != null && clickedSquare.getPiece() != null)
                {
                    if (clickedSquare.getPiece().getRank() == 0 || clickedSquare.getPiece().getRank() == 11)
                    {
                        //do nothing because the piece cannot move
                    }
                    else if (clickedSquare.getPiece().getRank() == 2)
                    {

                    }
                    else
                    {

                    }
                }
            }
        }
    }

    public delegate void displayPieceDelegate(object sender, EventArgs e);

    public void displayPiece(object sender, EventArgs e)
    {
        if (_gameLogic.isSoloGame() == true)
        {
            _displayTimer.Stop();
            int outCome = _gameLogic.processMove(_lastClickedSquare, _clickedSquare); //0 = your piece wins, 1 = their piece wins, 2 = tie

            //if the game has not ended
            if (_gameLogic.hasGameEnded() == false)
            {
                //if it is your turn
                if (_player.Equals(_gameLogic.getPlayers()[0]))
                {
                    //your piece moved to the enemy spot
                    if (outCome == 0)
                    {
                        _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                        _piecesOnScreen[_clickedSquare.getIndex() / 10][_clickedSquare.getIndex() % 10] = 1;
                    }
                    //your piece got killed and enemy piece stays
                    else if (outCome == 1)
                    {
                        _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                        _piecesOnScreen[_clickedSquare.getIndex() / 10][_clickedSquare.getIndex() % 10] = 2;
                    }
                    //both pieces are killed
                    else if (outCome == 2)
                    {
                        _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                        _piecesOnScreen[_clickedSquare.getIndex() / 10][_clickedSquare.getIndex() % 10] = 0;
                    }
                }
                //if it is not your turn
                else
                {
                    //enemy piece moved to your spot
                    if (outCome == 0)
                    {
                        _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                        _piecesOnScreen[_clickedSquare.getIndex() / 10][_clickedSquare.getIndex() % 10] = 2;
                    }
                    //enemy piece got killed and your piece stays
                    else if (outCome == 1)
                    {
                        _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                    }
                    //both pieces are killed
                    else if (outCome == 2)
                    {
                        _piecesOnScreen[_lastClickedSquare.getIndex() / 10][_lastClickedSquare.getIndex() % 10] = 0;
                        _piecesOnScreen[_clickedSquare.getIndex() / 10][_clickedSquare.getIndex() % 10] = 0;
                    }
                }
            }            

            _lastClickedSquare = null;
            _attackedSquare = null;
            _clickedSquare = null;
            _boardPanel.Invalidate();
            _graveyardPanel.Invalidate();
            _gameLogic.nextTurn();
        }
        //if the game is multiplayer
        else
        {
            _lastClickedSquare = null;
            _attackedSquare = null;
            _clickedSquare = null;

            _gameLogic.updateGame();

            _boardPanel.Invalidate();
            _graveyardPanel.Invalidate();
        }
    }

    private void menuItemClicked(object sender, EventArgs e)
    {
        if (_lastClickedSquare != null && _player.Equals(_gameLogic.getCurrentPlayer()) && _gameLogic.getPhase() == 0)
        {
            _lastClickedSquare = null;

            if (_gameLogic.getPhase() == 0)
            {
                _setUpPieces[_pieceSelected]--;
            }
        }
        if (_gameLogic.getPhase() == 0 && _pieceSelected < 12 && _player.Equals(_gameLogic.getCurrentPlayer()))
        {
            _pieceSelected = 12;
        }

        if (((MenuItem)sender).Text.CompareTo("Finalize Set-up") == 0)
        {
            //if all their pieces have been placed
            if (_emptySquares.Count == 0)
            {
                for (int x = 0; x < _board.getBoardSquares().Length; x++)
                {
                    for (int y = 0; y < _board.getBoardSquares()[x].Length; y++)
                    {
                        if (_board.getBoardSquares()[x][y].getPiece() != null && _board.getBoardSquares()[x][y].getPiece().getPlayer().Equals(_player))
                        {
                            _player.getPieces().Add(_board.getBoardSquares()[x][y].getPiece());
                        }
                    }
                }

                //if the player is able to make a move off the start
                if (_gameLogic.unableToMoveAtStart() == false)
                {
                    //if the game is single player
                    if (_gameLogic.isSoloGame() == true)
                    {
                        _gameLogic.startGame();
                    }
                    //if the game is multiplayer
                    else
                    {
                        //if the player wants to finalize their pieces
                        if (MessageBox.Show(_gameForm, "If you confirm you will not be able to change your pieces", "Confirm", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
                        {
                            _setUpMenu.Enabled = false;
                            if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                            {
                                _gameLogic.getStrategoClient().writeFinalized(_board.getBoardSquares(), _player);
                            }
                        }
                    }
                }
            }
        }
        else if (((MenuItem)sender).Text.CompareTo("Display 1-9") == 0)
        {
            _diplayOneThroughNineMenuItem.Text = "Display 2-10";
            _oneThroughNine = true;
        }
        else if (((MenuItem)sender).Text.CompareTo("Display 2-10") == 0)
        {
            _diplayOneThroughNineMenuItem.Text = "Display 1-9";
            _oneThroughNine = false;
        }
        else if (((MenuItem)sender).Text.CompareTo("Invert Display") == 0)
        {
            if (_invertDisplayMenuItem.Checked == true)
            {
                //if the player is the blue player
                if (_player.getPlayerColor().Equals(Color.Blue))
                {
                    int indexX = 0;
                    for (int x = _boardSquares.Length - 1; x >= 0; x--)
                    {
                        int indexY = 0;
                        for (int y = _boardSquares[x].Length - 1; y >= 0; y--)
                        {
                            _boardSquares[x][y].Location = new Point((1 + (indexY * _boardSquares[x][y].Width)), (2 + (indexX * _boardSquares[x][y].Height)));
                            indexY++;
                        }
                        indexX++;
                    }
                }
                //if the player is the red player
                else
                {
                    for (int x = 0; x < _boardSquares.Length; x++)
                    {
                        for (int y = 0; y < _boardSquares.Length; y++)
                        {
                            _boardSquares[x][y].Location = new Point((1 + (y * _boardSquares[x][y].Width)), (2 + (x * _boardSquares[x][y].Height)));
                        }
                    }
                }
            }
            else
            {
                //if the player is the blue player
                if (_player.getPlayerColor().Equals(Color.Blue))
                {
                    for (int x = 0; x < _boardSquares.Length; x++)
                    {
                        for (int y = 0; y < _boardSquares.Length; y++)
                        {
                            _boardSquares[x][y].Location = new Point((1 + (y * _boardSquares[x][y].Width)), (2 + (x * _boardSquares[x][y].Height)));
                        }
                    }

                }
                //if the player is the red player
                else
                {
                    int indexX = 0;
                    for (int x = _boardSquares.Length - 1; x >= 0; x--)
                    {
                        int indexY = 0;
                        for (int y = _boardSquares[x].Length - 1; y >= 0; y--)
                        {
                            _boardSquares[x][y].Location = new Point((1 + (indexY * _boardSquares[x][y].Width)), (2 + (indexX * _boardSquares[x][y].Height)));
                            indexY++;
                        }
                        indexX++;
                    }
                }
            }
            _invertDisplayMenuItem.Checked = !_invertDisplayMenuItem.Checked;
        }
        else if (((MenuItem)sender).Text.CompareTo("Propose Draw") == 0)
        {
            //if you are currently agreeing to a draw
            if (_drawProposalMenuItem.Checked == true)
            {
                //if the game is single player
                if (_gameLogic.isSoloGame() == true)
                {

                }
                //if the game is multiplayer
                else
                {
                    if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                    {
                        _gameLogic.getStrategoClient().proposeDraw(0);
                    }
                }
            }
            //if you are not currently agreeing to a draw
            else
            {
                //if the game is single player
                if (_gameLogic.isSoloGame() == true)
                {

                }
                //if the game is multiplayer
                else
                {
                    if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                    {
                        _gameLogic.getStrategoClient().proposeDraw(1);
                    }
                }
            }
            _drawProposalMenuItem.Checked = !_drawProposalMenuItem.Checked;
        }
        else if (((MenuItem)sender).Text.CompareTo("Show Pieces at End") == 0)
        {
            //if you already want to show your pieces at the end of the game
            if (_showPiecesAtEndMenuItem.Checked == true)
            {
                //if the game is multiplayer
                if (_gameLogic.isSoloGame() == false)
                {
                    if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                    {
                        _gameLogic.getStrategoClient().showPiecesAtend(0);
                    }
                }
            }
            //if you do not already want to show your pieces at the end of the game
            else
            {
                //if the game is multiplayer
                if (_gameLogic.isSoloGame() == false)
                {
                    if (_gameLogic.getStrategoClient().isSocketConnected() == true)
                    {
                        _gameLogic.getStrategoClient().showPiecesAtend(1);
                    }
                }
            }
            _showPiecesAtEndMenuItem.Checked = !_showPiecesAtEndMenuItem.Checked;
        }
        else if (((MenuItem)sender).Text.CompareTo("Random Set-up") == 0)
        {
            ArrayList randomNumbers = new ArrayList(_setUpPieces.Length);
            for (int x = 0; x < _setUpPieces.Length; x++)
            {
                if (_setUpPieces[x] > 0)
                {
                    randomNumbers.Add(x);
                }
            }
            Random rand = new Random();
            while (_emptySquares.Count > 0)
            {
                int num = rand.Next(randomNumbers.Count), square = rand.Next(_emptySquares.Count);
                ((BoardSquare)_emptySquares[square]).setPiece(new Piece(((int)randomNumbers[num]), _player, ((BoardSquare)_emptySquares[square])));
                _setUpPieces[((int)randomNumbers[num])]--;
                _emptySquares.RemoveAt(square);
                if (_setUpPieces[((int)randomNumbers[num])] == 0)
                {
                    randomNumbers.RemoveAt(num);
                }
            }
        }
        else if (((MenuItem)sender).Text.CompareTo("Set-up #1") == 0)
        {
            if (_player.getPlayerColor().Equals(Color.Blue))
            {
                _board.getBoardSquares()[3][9].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][9]));
                _board.getBoardSquares()[3][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[3][8]));
                _board.getBoardSquares()[3][7].setPiece(new Piece(4, _player, _board.getBoardSquares()[3][7]));
                _board.getBoardSquares()[3][6].setPiece(new Piece(0, _player, _board.getBoardSquares()[3][6]));
                _board.getBoardSquares()[3][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][5]));
                _board.getBoardSquares()[3][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][4]));
                _board.getBoardSquares()[3][3].setPiece(new Piece(3, _player, _board.getBoardSquares()[3][3]));
                _board.getBoardSquares()[3][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][2]));
                _board.getBoardSquares()[3][1].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][1]));
                _board.getBoardSquares()[3][0].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][0]));
                _board.getBoardSquares()[2][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[2][9]));
                _board.getBoardSquares()[2][8].setPiece(new Piece(4, _player, _board.getBoardSquares()[2][8]));
                _board.getBoardSquares()[2][7].setPiece(new Piece(0, _player, _board.getBoardSquares()[2][7]));
                _board.getBoardSquares()[2][6].setPiece(new Piece(10, _player, _board.getBoardSquares()[2][6]));
                _board.getBoardSquares()[2][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[2][5]));
                _board.getBoardSquares()[2][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][4]));
                _board.getBoardSquares()[2][3].setPiece(new Piece(9, _player, _board.getBoardSquares()[2][3]));
                _board.getBoardSquares()[2][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][2]));
                _board.getBoardSquares()[2][1].setPiece(new Piece(3, _player, _board.getBoardSquares()[2][1]));
                _board.getBoardSquares()[2][0].setPiece(new Piece(8, _player, _board.getBoardSquares()[2][0]));
                _board.getBoardSquares()[1][9].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][9]));
                _board.getBoardSquares()[1][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][8]));
                _board.getBoardSquares()[1][7].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][7]));
                _board.getBoardSquares()[1][6].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][6]));
                _board.getBoardSquares()[1][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][5]));
                _board.getBoardSquares()[1][4].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][4]));
                _board.getBoardSquares()[1][3].setPiece(new Piece(1, _player, _board.getBoardSquares()[1][3]));
                _board.getBoardSquares()[1][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][2]));
                _board.getBoardSquares()[1][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][1]));
                _board.getBoardSquares()[1][0].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][0]));
                _board.getBoardSquares()[0][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][9]));
                _board.getBoardSquares()[0][8].setPiece(new Piece(3, _player, _board.getBoardSquares()[0][8]));
                _board.getBoardSquares()[0][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[0][7]));
                _board.getBoardSquares()[0][6].setPiece(new Piece(11, _player, _board.getBoardSquares()[0][6]));
                _board.getBoardSquares()[0][5].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][5]));
                _board.getBoardSquares()[0][4].setPiece(new Piece(8, _player, _board.getBoardSquares()[0][4]));
                _board.getBoardSquares()[0][3].setPiece(new Piece(6, _player, _board.getBoardSquares()[0][3]));
                _board.getBoardSquares()[0][2].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][2]));
                _board.getBoardSquares()[0][1].setPiece(new Piece(6, _player, _board.getBoardSquares()[0][1]));
                _board.getBoardSquares()[0][0].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][0]));
            }
            else
            {
                _board.getBoardSquares()[6][0].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][0]));
                _board.getBoardSquares()[6][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[6][1]));
                _board.getBoardSquares()[6][2].setPiece(new Piece(4, _player, _board.getBoardSquares()[6][2]));
                _board.getBoardSquares()[6][3].setPiece(new Piece(0, _player, _board.getBoardSquares()[6][3]));
                _board.getBoardSquares()[6][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][4]));
                _board.getBoardSquares()[6][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][5]));
                _board.getBoardSquares()[6][6].setPiece(new Piece(3, _player, _board.getBoardSquares()[6][6]));
                _board.getBoardSquares()[6][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][7]));
                _board.getBoardSquares()[6][8].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][8]));
                _board.getBoardSquares()[6][9].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][9]));
                _board.getBoardSquares()[7][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[7][0]));
                _board.getBoardSquares()[7][1].setPiece(new Piece(4, _player, _board.getBoardSquares()[7][1]));
                _board.getBoardSquares()[7][2].setPiece(new Piece(0, _player, _board.getBoardSquares()[7][2]));
                _board.getBoardSquares()[7][3].setPiece(new Piece(10, _player, _board.getBoardSquares()[7][3]));
                _board.getBoardSquares()[7][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[7][4]));
                _board.getBoardSquares()[7][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][5]));
                _board.getBoardSquares()[7][6].setPiece(new Piece(9, _player, _board.getBoardSquares()[7][6]));
                _board.getBoardSquares()[7][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][7]));
                _board.getBoardSquares()[7][8].setPiece(new Piece(3, _player, _board.getBoardSquares()[7][8]));
                _board.getBoardSquares()[7][9].setPiece(new Piece(8, _player, _board.getBoardSquares()[7][9]));
                _board.getBoardSquares()[8][0].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][0]));
                _board.getBoardSquares()[8][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][1]));
                _board.getBoardSquares()[8][2].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][2]));
                _board.getBoardSquares()[8][3].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][3]));
                _board.getBoardSquares()[8][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][4]));
                _board.getBoardSquares()[8][5].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][5]));
                _board.getBoardSquares()[8][6].setPiece(new Piece(1, _player, _board.getBoardSquares()[8][6]));
                _board.getBoardSquares()[8][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][7]));
                _board.getBoardSquares()[8][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][8]));
                _board.getBoardSquares()[8][9].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][9]));
                _board.getBoardSquares()[9][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][0]));
                _board.getBoardSquares()[9][1].setPiece(new Piece(3, _player, _board.getBoardSquares()[9][1]));
                _board.getBoardSquares()[9][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[9][2]));
                _board.getBoardSquares()[9][3].setPiece(new Piece(11, _player, _board.getBoardSquares()[9][3]));
                _board.getBoardSquares()[9][4].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][4]));
                _board.getBoardSquares()[9][5].setPiece(new Piece(8, _player, _board.getBoardSquares()[9][5]));
                _board.getBoardSquares()[9][6].setPiece(new Piece(6, _player, _board.getBoardSquares()[9][6]));
                _board.getBoardSquares()[9][7].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][7]));
                _board.getBoardSquares()[9][8].setPiece(new Piece(6, _player, _board.getBoardSquares()[9][8]));
                _board.getBoardSquares()[9][9].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][9]));
            }
            for (int x = 0; x < _setUpPieces.Length; x++)
            {
                _setUpPieces[x] = 0;
            }
            _emptySquares = new ArrayList(40);
        }
        else if (((MenuItem)sender).Text.CompareTo("Set-up #2") == 0)
        {
            if (_player.getPlayerColor().Equals(Color.Blue))
            {
                _board.getBoardSquares()[3][9].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][9]));
                _board.getBoardSquares()[3][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[3][8]));
                _board.getBoardSquares()[3][7].setPiece(new Piece(4, _player, _board.getBoardSquares()[3][7]));
                _board.getBoardSquares()[3][6].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][6]));
                _board.getBoardSquares()[3][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][5]));
                _board.getBoardSquares()[3][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][4]));
                _board.getBoardSquares()[3][3].setPiece(new Piece(3, _player, _board.getBoardSquares()[3][3]));
                _board.getBoardSquares()[3][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][2]));
                _board.getBoardSquares()[3][1].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][1]));
                _board.getBoardSquares()[3][0].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][0]));
                _board.getBoardSquares()[2][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[2][9]));
                _board.getBoardSquares()[2][8].setPiece(new Piece(4, _player, _board.getBoardSquares()[2][8]));
                _board.getBoardSquares()[2][7].setPiece(new Piece(0, _player, _board.getBoardSquares()[2][7]));
                _board.getBoardSquares()[2][6].setPiece(new Piece(9, _player, _board.getBoardSquares()[2][6]));
                _board.getBoardSquares()[2][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[2][5]));
                _board.getBoardSquares()[2][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][4]));
                _board.getBoardSquares()[2][3].setPiece(new Piece(10, _player, _board.getBoardSquares()[2][3]));
                _board.getBoardSquares()[2][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][2]));
                _board.getBoardSquares()[2][1].setPiece(new Piece(3, _player, _board.getBoardSquares()[2][1]));
                _board.getBoardSquares()[2][0].setPiece(new Piece(8, _player, _board.getBoardSquares()[2][0]));
                _board.getBoardSquares()[1][9].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][9]));
                _board.getBoardSquares()[1][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][8]));
                _board.getBoardSquares()[1][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[1][7]));
                _board.getBoardSquares()[1][6].setPiece(new Piece(1, _player, _board.getBoardSquares()[1][6]));
                _board.getBoardSquares()[1][5].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][5]));
                _board.getBoardSquares()[1][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][4]));
                _board.getBoardSquares()[1][3].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][3]));
                _board.getBoardSquares()[1][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[1][2]));
                _board.getBoardSquares()[1][1].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][1]));
                _board.getBoardSquares()[1][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][0]));
                _board.getBoardSquares()[0][9].setPiece(new Piece(2, _player, _board.getBoardSquares()[0][9]));
                _board.getBoardSquares()[0][8].setPiece(new Piece(3, _player, _board.getBoardSquares()[0][8]));
                _board.getBoardSquares()[0][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[0][7]));
                _board.getBoardSquares()[0][6].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][6]));
                _board.getBoardSquares()[0][5].setPiece(new Piece(8, _player, _board.getBoardSquares()[0][5]));
                _board.getBoardSquares()[0][4].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][4]));
                _board.getBoardSquares()[0][3].setPiece(new Piece(3, _player, _board.getBoardSquares()[0][3]));
                _board.getBoardSquares()[0][2].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][2]));
                _board.getBoardSquares()[0][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][1]));
                _board.getBoardSquares()[0][0].setPiece(new Piece(11, _player, _board.getBoardSquares()[0][0]));
            }
            else
            {
                _board.getBoardSquares()[6][0].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][0]));
                _board.getBoardSquares()[6][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[6][1]));
                _board.getBoardSquares()[6][2].setPiece(new Piece(4, _player, _board.getBoardSquares()[6][2]));
                _board.getBoardSquares()[6][3].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][3]));
                _board.getBoardSquares()[6][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][4]));
                _board.getBoardSquares()[6][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][5]));
                _board.getBoardSquares()[6][6].setPiece(new Piece(3, _player, _board.getBoardSquares()[6][6]));
                _board.getBoardSquares()[6][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][7]));
                _board.getBoardSquares()[6][8].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][8]));
                _board.getBoardSquares()[6][9].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][9]));
                _board.getBoardSquares()[7][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[7][0]));
                _board.getBoardSquares()[7][1].setPiece(new Piece(4, _player, _board.getBoardSquares()[7][1]));
                _board.getBoardSquares()[7][2].setPiece(new Piece(0, _player, _board.getBoardSquares()[7][2]));
                _board.getBoardSquares()[7][3].setPiece(new Piece(9, _player, _board.getBoardSquares()[7][3]));
                _board.getBoardSquares()[7][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[7][4]));
                _board.getBoardSquares()[7][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][5]));
                _board.getBoardSquares()[7][6].setPiece(new Piece(10, _player, _board.getBoardSquares()[7][6]));
                _board.getBoardSquares()[7][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][7]));
                _board.getBoardSquares()[7][8].setPiece(new Piece(3, _player, _board.getBoardSquares()[7][8]));
                _board.getBoardSquares()[7][9].setPiece(new Piece(8, _player, _board.getBoardSquares()[7][9]));
                _board.getBoardSquares()[8][0].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][0]));
                _board.getBoardSquares()[8][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][1]));
                _board.getBoardSquares()[8][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[8][2]));
                _board.getBoardSquares()[8][3].setPiece(new Piece(1, _player, _board.getBoardSquares()[8][3]));
                _board.getBoardSquares()[8][4].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][4]));
                _board.getBoardSquares()[8][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][5]));
                _board.getBoardSquares()[8][6].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][6]));
                _board.getBoardSquares()[8][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[8][7]));
                _board.getBoardSquares()[8][8].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][8]));
                _board.getBoardSquares()[8][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][9]));
                _board.getBoardSquares()[9][0].setPiece(new Piece(2, _player, _board.getBoardSquares()[9][0]));
                _board.getBoardSquares()[9][1].setPiece(new Piece(3, _player, _board.getBoardSquares()[9][1]));
                _board.getBoardSquares()[9][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[9][2]));
                _board.getBoardSquares()[9][3].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][3]));
                _board.getBoardSquares()[9][4].setPiece(new Piece(8, _player, _board.getBoardSquares()[9][4]));
                _board.getBoardSquares()[9][5].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][5]));
                _board.getBoardSquares()[9][6].setPiece(new Piece(3, _player, _board.getBoardSquares()[9][6]));
                _board.getBoardSquares()[9][7].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][7]));
                _board.getBoardSquares()[9][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][8]));
                _board.getBoardSquares()[9][9].setPiece(new Piece(11, _player, _board.getBoardSquares()[9][9]));
            }
            for (int x = 0; x < _setUpPieces.Length; x++)
            {
                _setUpPieces[x] = 0;
            }
            _emptySquares = new ArrayList(40);
        }
        else if (((MenuItem)sender).Text.CompareTo("Set-up #3") == 0)
        {
            if (_player.getPlayerColor().Equals(Color.Blue))
            {
                _board.getBoardSquares()[3][9].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][9]));
                _board.getBoardSquares()[3][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][8]));
                _board.getBoardSquares()[3][7].setPiece(new Piece(0, _player, _board.getBoardSquares()[3][7]));
                _board.getBoardSquares()[3][6].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][6]));
                _board.getBoardSquares()[3][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][5]));
                _board.getBoardSquares()[3][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][4]));
                _board.getBoardSquares()[3][3].setPiece(new Piece(3, _player, _board.getBoardSquares()[3][3]));
                _board.getBoardSquares()[3][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][2]));
                _board.getBoardSquares()[3][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][1]));
                _board.getBoardSquares()[3][0].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][0]));
                _board.getBoardSquares()[2][9].setPiece(new Piece(4, _player, _board.getBoardSquares()[2][9]));
                _board.getBoardSquares()[2][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[2][8]));
                _board.getBoardSquares()[2][7].setPiece(new Piece(4, _player, _board.getBoardSquares()[2][7]));
                _board.getBoardSquares()[2][6].setPiece(new Piece(10, _player, _board.getBoardSquares()[2][6]));
                _board.getBoardSquares()[2][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][5]));
                _board.getBoardSquares()[2][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[2][4]));
                _board.getBoardSquares()[2][3].setPiece(new Piece(9, _player, _board.getBoardSquares()[2][3]));
                _board.getBoardSquares()[2][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][2]));
                _board.getBoardSquares()[2][1].setPiece(new Piece(3, _player, _board.getBoardSquares()[2][1]));
                _board.getBoardSquares()[2][0].setPiece(new Piece(8, _player, _board.getBoardSquares()[2][0]));
                _board.getBoardSquares()[1][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][9]));
                _board.getBoardSquares()[1][8].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][8]));
                _board.getBoardSquares()[1][7].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][7]));
                _board.getBoardSquares()[1][6].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][6]));
                _board.getBoardSquares()[1][5].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][5]));
                _board.getBoardSquares()[1][4].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][4]));
                _board.getBoardSquares()[1][3].setPiece(new Piece(1, _player, _board.getBoardSquares()[1][3]));
                _board.getBoardSquares()[1][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][2]));
                _board.getBoardSquares()[1][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][1]));
                _board.getBoardSquares()[1][0].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][0]));
                _board.getBoardSquares()[0][9].setPiece(new Piece(6, _player, _board.getBoardSquares()[0][9]));
                _board.getBoardSquares()[0][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[0][8]));
                _board.getBoardSquares()[0][7].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][7]));
                _board.getBoardSquares()[0][6].setPiece(new Piece(11, _player, _board.getBoardSquares()[0][6]));
                _board.getBoardSquares()[0][5].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][5]));
                _board.getBoardSquares()[0][4].setPiece(new Piece(8, _player, _board.getBoardSquares()[0][4]));
                _board.getBoardSquares()[0][3].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][3]));
                _board.getBoardSquares()[0][2].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][2]));
                _board.getBoardSquares()[0][1].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][1]));
                _board.getBoardSquares()[0][0].setPiece(new Piece(6, _player, _board.getBoardSquares()[0][0]));
            }
            else
            {
                _board.getBoardSquares()[6][0].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][0]));
                _board.getBoardSquares()[6][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][1]));
                _board.getBoardSquares()[6][2].setPiece(new Piece(0, _player, _board.getBoardSquares()[6][2]));
                _board.getBoardSquares()[6][3].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][3]));
                _board.getBoardSquares()[6][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][4]));
                _board.getBoardSquares()[6][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][5]));
                _board.getBoardSquares()[6][6].setPiece(new Piece(3, _player, _board.getBoardSquares()[6][6]));
                _board.getBoardSquares()[6][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][7]));
                _board.getBoardSquares()[6][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][8]));
                _board.getBoardSquares()[6][9].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][9]));
                _board.getBoardSquares()[7][0].setPiece(new Piece(4, _player, _board.getBoardSquares()[7][0]));
                _board.getBoardSquares()[7][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[7][1]));
                _board.getBoardSquares()[7][2].setPiece(new Piece(4, _player, _board.getBoardSquares()[7][2]));
                _board.getBoardSquares()[7][3].setPiece(new Piece(10, _player, _board.getBoardSquares()[7][3]));
                _board.getBoardSquares()[7][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][4]));
                _board.getBoardSquares()[7][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[7][5]));
                _board.getBoardSquares()[7][6].setPiece(new Piece(9, _player, _board.getBoardSquares()[7][6]));
                _board.getBoardSquares()[7][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][7]));
                _board.getBoardSquares()[7][8].setPiece(new Piece(3, _player, _board.getBoardSquares()[7][8]));
                _board.getBoardSquares()[7][9].setPiece(new Piece(8, _player, _board.getBoardSquares()[7][9]));
                _board.getBoardSquares()[8][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][0]));
                _board.getBoardSquares()[8][1].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][1]));
                _board.getBoardSquares()[8][2].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][2]));
                _board.getBoardSquares()[8][3].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][3]));
                _board.getBoardSquares()[8][4].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][4]));
                _board.getBoardSquares()[8][5].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][5]));
                _board.getBoardSquares()[8][6].setPiece(new Piece(1, _player, _board.getBoardSquares()[8][6]));
                _board.getBoardSquares()[8][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][7]));
                _board.getBoardSquares()[8][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][8]));
                _board.getBoardSquares()[8][9].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][9]));
                _board.getBoardSquares()[9][0].setPiece(new Piece(6, _player, _board.getBoardSquares()[9][0]));
                _board.getBoardSquares()[9][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[9][1]));
                _board.getBoardSquares()[9][2].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][2]));
                _board.getBoardSquares()[9][3].setPiece(new Piece(11, _player, _board.getBoardSquares()[9][3]));
                _board.getBoardSquares()[9][4].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][4]));
                _board.getBoardSquares()[9][5].setPiece(new Piece(8, _player, _board.getBoardSquares()[9][5]));
                _board.getBoardSquares()[9][6].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][6]));
                _board.getBoardSquares()[9][7].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][7]));
                _board.getBoardSquares()[9][8].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][8]));
                _board.getBoardSquares()[9][9].setPiece(new Piece(6, _player, _board.getBoardSquares()[9][9]));
            }
            for (int x = 0; x < _setUpPieces.Length; x++)
            {
                _setUpPieces[x] = 0;
            }
            _emptySquares = new ArrayList(40);
        }
        else if (((MenuItem)sender).Text.CompareTo("Set-up #4") == 0)
        {
            if (_player.getPlayerColor().Equals(Color.Blue))
            {
                _board.getBoardSquares()[3][9].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][9]));
                _board.getBoardSquares()[3][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][8]));
                _board.getBoardSquares()[3][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][7]));
                _board.getBoardSquares()[3][6].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][6]));
                _board.getBoardSquares()[3][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][5]));
                _board.getBoardSquares()[3][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][4]));
                _board.getBoardSquares()[3][3].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][3]));
                _board.getBoardSquares()[3][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][2]));
                _board.getBoardSquares()[3][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][1]));
                _board.getBoardSquares()[3][0].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][0]));
                _board.getBoardSquares()[2][9].setPiece(new Piece(8, _player, _board.getBoardSquares()[2][9]));
                _board.getBoardSquares()[2][8].setPiece(new Piece(3, _player, _board.getBoardSquares()[2][8]));
                _board.getBoardSquares()[2][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][7]));
                _board.getBoardSquares()[2][6].setPiece(new Piece(10, _player, _board.getBoardSquares()[2][6]));
                _board.getBoardSquares()[2][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[2][5]));
                _board.getBoardSquares()[2][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][4]));
                _board.getBoardSquares()[2][3].setPiece(new Piece(9, _player, _board.getBoardSquares()[2][3]));
                _board.getBoardSquares()[2][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][2]));
                _board.getBoardSquares()[2][1].setPiece(new Piece(3, _player, _board.getBoardSquares()[2][1]));
                _board.getBoardSquares()[2][0].setPiece(new Piece(8, _player, _board.getBoardSquares()[2][0]));
                _board.getBoardSquares()[1][9].setPiece(new Piece(7, _player, _board.getBoardSquares()[1][9]));
                _board.getBoardSquares()[1][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][8]));
                _board.getBoardSquares()[1][7].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][7]));
                _board.getBoardSquares()[1][6].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][6]));
                _board.getBoardSquares()[1][5].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][5]));
                _board.getBoardSquares()[1][4].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][4]));
                _board.getBoardSquares()[1][3].setPiece(new Piece(1, _player, _board.getBoardSquares()[1][3]));
                _board.getBoardSquares()[1][2].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][2]));
                _board.getBoardSquares()[1][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][1]));
                _board.getBoardSquares()[1][0].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][0]));
                _board.getBoardSquares()[0][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][9]));
                _board.getBoardSquares()[0][8].setPiece(new Piece(4, _player, _board.getBoardSquares()[0][8]));
                _board.getBoardSquares()[0][7].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][7]));
                _board.getBoardSquares()[0][6].setPiece(new Piece(11, _player, _board.getBoardSquares()[0][6]));
                _board.getBoardSquares()[0][5].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][5]));
                _board.getBoardSquares()[0][4].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][4]));
                _board.getBoardSquares()[0][3].setPiece(new Piece(3, _player, _board.getBoardSquares()[0][3]));
                _board.getBoardSquares()[0][2].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][2]));
                _board.getBoardSquares()[0][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][1]));
                _board.getBoardSquares()[0][0].setPiece(new Piece(4, _player, _board.getBoardSquares()[0][0]));
            }
            else
            {
                _board.getBoardSquares()[6][0].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][0]));
                _board.getBoardSquares()[6][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][1]));
                _board.getBoardSquares()[6][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][2]));
                _board.getBoardSquares()[6][3].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][3]));
                _board.getBoardSquares()[6][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][4]));
                _board.getBoardSquares()[6][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][5]));
                _board.getBoardSquares()[6][6].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][6]));
                _board.getBoardSquares()[6][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][7]));
                _board.getBoardSquares()[6][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][8]));
                _board.getBoardSquares()[6][9].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][9]));
                _board.getBoardSquares()[7][0].setPiece(new Piece(8, _player, _board.getBoardSquares()[7][0]));
                _board.getBoardSquares()[7][1].setPiece(new Piece(3, _player, _board.getBoardSquares()[7][1]));
                _board.getBoardSquares()[7][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][2]));
                _board.getBoardSquares()[7][3].setPiece(new Piece(10, _player, _board.getBoardSquares()[7][3]));
                _board.getBoardSquares()[7][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[7][4]));
                _board.getBoardSquares()[7][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][5]));
                _board.getBoardSquares()[7][6].setPiece(new Piece(9, _player, _board.getBoardSquares()[7][6]));
                _board.getBoardSquares()[7][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][7]));
                _board.getBoardSquares()[7][8].setPiece(new Piece(3, _player, _board.getBoardSquares()[7][8]));
                _board.getBoardSquares()[7][9].setPiece(new Piece(8, _player, _board.getBoardSquares()[7][9]));
                _board.getBoardSquares()[8][0].setPiece(new Piece(7, _player, _board.getBoardSquares()[8][0]));
                _board.getBoardSquares()[8][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][1]));
                _board.getBoardSquares()[8][2].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][2]));
                _board.getBoardSquares()[8][3].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][3]));
                _board.getBoardSquares()[8][4].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][4]));
                _board.getBoardSquares()[8][5].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][5]));
                _board.getBoardSquares()[8][6].setPiece(new Piece(1, _player, _board.getBoardSquares()[8][6]));
                _board.getBoardSquares()[8][7].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][7]));
                _board.getBoardSquares()[8][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][8]));
                _board.getBoardSquares()[8][9].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][9]));
                _board.getBoardSquares()[9][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][0]));
                _board.getBoardSquares()[9][1].setPiece(new Piece(4, _player, _board.getBoardSquares()[9][1]));
                _board.getBoardSquares()[9][2].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][2]));
                _board.getBoardSquares()[9][3].setPiece(new Piece(11, _player, _board.getBoardSquares()[9][3]));
                _board.getBoardSquares()[9][4].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][4]));
                _board.getBoardSquares()[9][5].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][5]));
                _board.getBoardSquares()[9][6].setPiece(new Piece(3, _player, _board.getBoardSquares()[9][6]));
                _board.getBoardSquares()[9][7].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][7]));
                _board.getBoardSquares()[9][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][8]));
                _board.getBoardSquares()[9][9].setPiece(new Piece(4, _player, _board.getBoardSquares()[9][9]));
            }
            for (int x = 0; x < _setUpPieces.Length; x++)
            {
                _setUpPieces[x] = 0;
            }
            _emptySquares = new ArrayList(40);
        }
        else if (((MenuItem)sender).Text.CompareTo("Set-up #5") == 0)
        {
            if (_player.getPlayerColor().Equals(Color.Blue))
            {
                _board.getBoardSquares()[3][9].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][9]));
                _board.getBoardSquares()[3][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[3][8]));
                _board.getBoardSquares()[3][7].setPiece(new Piece(4, _player, _board.getBoardSquares()[3][7]));
                _board.getBoardSquares()[3][6].setPiece(new Piece(3, _player, _board.getBoardSquares()[3][6]));
                _board.getBoardSquares()[3][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][5]));
                _board.getBoardSquares()[3][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][4]));
                _board.getBoardSquares()[3][3].setPiece(new Piece(3, _player, _board.getBoardSquares()[3][3]));
                _board.getBoardSquares()[3][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[3][2]));
                _board.getBoardSquares()[3][1].setPiece(new Piece(5, _player, _board.getBoardSquares()[3][1]));
                _board.getBoardSquares()[3][0].setPiece(new Piece(2, _player, _board.getBoardSquares()[3][0]));
                _board.getBoardSquares()[2][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[2][9]));
                _board.getBoardSquares()[2][8].setPiece(new Piece(4, _player, _board.getBoardSquares()[2][8]));
                _board.getBoardSquares()[2][7].setPiece(new Piece(0, _player, _board.getBoardSquares()[2][7]));
                _board.getBoardSquares()[2][6].setPiece(new Piece(10, _player, _board.getBoardSquares()[2][6]));
                _board.getBoardSquares()[2][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[2][5]));
                _board.getBoardSquares()[2][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][4]));
                _board.getBoardSquares()[2][3].setPiece(new Piece(9, _player, _board.getBoardSquares()[2][3]));
                _board.getBoardSquares()[2][2].setPiece(new Piece(6, _player, _board.getBoardSquares()[2][2]));
                _board.getBoardSquares()[2][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[2][1]));
                _board.getBoardSquares()[2][0].setPiece(new Piece(8, _player, _board.getBoardSquares()[2][0]));
                _board.getBoardSquares()[1][9].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][9]));
                _board.getBoardSquares()[1][8].setPiece(new Piece(0, _player, _board.getBoardSquares()[1][8]));
                _board.getBoardSquares()[1][7].setPiece(new Piece(4, _player, _board.getBoardSquares()[1][7]));
                _board.getBoardSquares()[1][6].setPiece(new Piece(6, _player, _board.getBoardSquares()[1][6]));
                _board.getBoardSquares()[1][5].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][5]));
                _board.getBoardSquares()[1][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][4]));
                _board.getBoardSquares()[1][3].setPiece(new Piece(1, _player, _board.getBoardSquares()[1][3]));
                _board.getBoardSquares()[1][2].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][2]));
                _board.getBoardSquares()[1][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[1][1]));
                _board.getBoardSquares()[1][0].setPiece(new Piece(3, _player, _board.getBoardSquares()[1][0]));
                _board.getBoardSquares()[0][9].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][9]));
                _board.getBoardSquares()[0][8].setPiece(new Piece(11, _player, _board.getBoardSquares()[0][8]));
                _board.getBoardSquares()[0][7].setPiece(new Piece(0, _player, _board.getBoardSquares()[0][7]));
                _board.getBoardSquares()[0][6].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][6]));
                _board.getBoardSquares()[0][5].setPiece(new Piece(8, _player, _board.getBoardSquares()[0][5]));
                _board.getBoardSquares()[0][4].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][4]));
                _board.getBoardSquares()[0][3].setPiece(new Piece(6, _player, _board.getBoardSquares()[0][3]));
                _board.getBoardSquares()[0][2].setPiece(new Piece(3, _player, _board.getBoardSquares()[0][2]));
                _board.getBoardSquares()[0][1].setPiece(new Piece(2, _player, _board.getBoardSquares()[0][1]));
                _board.getBoardSquares()[0][0].setPiece(new Piece(7, _player, _board.getBoardSquares()[0][0]));
            }
            else
            {
                _board.getBoardSquares()[6][0].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][0]));
                _board.getBoardSquares()[6][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[6][1]));
                _board.getBoardSquares()[6][2].setPiece(new Piece(4, _player, _board.getBoardSquares()[6][2]));
                _board.getBoardSquares()[6][3].setPiece(new Piece(3, _player, _board.getBoardSquares()[6][3]));
                _board.getBoardSquares()[6][4].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][4]));
                _board.getBoardSquares()[6][5].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][5]));
                _board.getBoardSquares()[6][6].setPiece(new Piece(3, _player, _board.getBoardSquares()[6][6]));
                _board.getBoardSquares()[6][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[6][7]));
                _board.getBoardSquares()[6][8].setPiece(new Piece(5, _player, _board.getBoardSquares()[6][8]));
                _board.getBoardSquares()[6][9].setPiece(new Piece(2, _player, _board.getBoardSquares()[6][9]));
                _board.getBoardSquares()[7][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[7][0]));
                _board.getBoardSquares()[7][1].setPiece(new Piece(4, _player, _board.getBoardSquares()[7][1]));
                _board.getBoardSquares()[7][2].setPiece(new Piece(0, _player, _board.getBoardSquares()[7][2]));
                _board.getBoardSquares()[7][3].setPiece(new Piece(10, _player, _board.getBoardSquares()[7][3]));
                _board.getBoardSquares()[7][4].setPiece(new Piece(5, _player, _board.getBoardSquares()[7][4]));
                _board.getBoardSquares()[7][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][5]));
                _board.getBoardSquares()[7][6].setPiece(new Piece(9, _player, _board.getBoardSquares()[7][6]));
                _board.getBoardSquares()[7][7].setPiece(new Piece(6, _player, _board.getBoardSquares()[7][7]));
                _board.getBoardSquares()[7][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[7][8]));
                _board.getBoardSquares()[7][9].setPiece(new Piece(8, _player, _board.getBoardSquares()[7][9]));
                _board.getBoardSquares()[8][0].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][0]));
                _board.getBoardSquares()[8][1].setPiece(new Piece(0, _player, _board.getBoardSquares()[8][1]));
                _board.getBoardSquares()[8][2].setPiece(new Piece(4, _player, _board.getBoardSquares()[8][2]));
                _board.getBoardSquares()[8][3].setPiece(new Piece(6, _player, _board.getBoardSquares()[8][3]));
                _board.getBoardSquares()[8][4].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][4]));
                _board.getBoardSquares()[8][5].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][5]));
                _board.getBoardSquares()[8][6].setPiece(new Piece(1, _player, _board.getBoardSquares()[8][6]));
                _board.getBoardSquares()[8][7].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][7]));
                _board.getBoardSquares()[8][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[8][8]));
                _board.getBoardSquares()[8][9].setPiece(new Piece(3, _player, _board.getBoardSquares()[8][9]));
                _board.getBoardSquares()[9][0].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][0]));
                _board.getBoardSquares()[9][1].setPiece(new Piece(11, _player, _board.getBoardSquares()[9][1]));
                _board.getBoardSquares()[9][2].setPiece(new Piece(0, _player, _board.getBoardSquares()[9][2]));
                _board.getBoardSquares()[9][3].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][3]));
                _board.getBoardSquares()[9][4].setPiece(new Piece(8, _player, _board.getBoardSquares()[9][4]));
                _board.getBoardSquares()[9][5].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][5]));
                _board.getBoardSquares()[9][6].setPiece(new Piece(6, _player, _board.getBoardSquares()[9][6]));
                _board.getBoardSquares()[9][7].setPiece(new Piece(3, _player, _board.getBoardSquares()[9][7]));
                _board.getBoardSquares()[9][8].setPiece(new Piece(2, _player, _board.getBoardSquares()[9][8]));
                _board.getBoardSquares()[9][9].setPiece(new Piece(7, _player, _board.getBoardSquares()[9][9]));
            }
            for (int x = 0; x < _setUpPieces.Length; x++)
            {
                _setUpPieces[x] = 0;
            }
            _emptySquares = new ArrayList(40);
        }
        else if (((MenuItem)sender).Text.CompareTo("Mirror") == 0)
        {
            for (int x = 0; x < _boardSquares.Length; x++)
            {
                int mirror = 9;
                for (int y = 0; y < (_boardSquares[x].Length / 2); y++)
                {
                    //if one of the squares is empty but the other is not
                    if (_board.getBoardSquares()[x][y].getPiece() == null && _board.getBoardSquares()[x][y + mirror].getPiece() != null)
                    {
                        _emptySquares.Remove(_board.getBoardSquares()[x][y]);
                        _emptySquares.Add(_board.getBoardSquares()[x][y + mirror]);
                    }
                    //if one of the squares is not empty but the other is
                    else if (_board.getBoardSquares()[x][y].getPiece() != null && _board.getBoardSquares()[x][y + mirror].getPiece() == null)
                    {
                        _emptySquares.Remove(_board.getBoardSquares()[x][y + mirror]);
                        _emptySquares.Add(_board.getBoardSquares()[x][y]);
                    }
                    Piece piece = _board.getBoardSquares()[x][y].getPiece();
                    _board.getBoardSquares()[x][y].setPiece(_board.getBoardSquares()[x][y + mirror].getPiece());
                    if (_board.getBoardSquares()[x][y].getPiece() != null)
                    {
                        _board.getBoardSquares()[x][y].getPiece().setCurrentSquare(_board.getBoardSquares()[x][y]);
                    }
                    _board.getBoardSquares()[x][y + mirror].setPiece(piece);
                    if (_board.getBoardSquares()[x][y + mirror].getPiece() != null)
                    {
                        _board.getBoardSquares()[x][y + mirror].getPiece().setCurrentSquare(_board.getBoardSquares()[x][y + mirror]);
                    }
                    mirror = mirror - 2;
                }
            }
        }
        else if (((MenuItem)sender).Text.CompareTo("Clear Pieces") == 0)
        {
            for (int x = 0; x < _boardSquares.Length; x++)
            {
                for (int y = 0; y < _boardSquares[x].Length; y++)
                {
                    if (_board.getBoardSquares()[x][y].getPiece() != null)
                    {
                        _board.getBoardSquares()[x][y].setPiece(null);
                        _emptySquares.Add(_board.getBoardSquares()[x][y]);
                    }
                }
            }
            _setUpPieces[0] = 6;
            _setUpPieces[1] = 1;
            _setUpPieces[2] = 8;
            _setUpPieces[3] = 5;
            _setUpPieces[4] = 4;
            _setUpPieces[5] = 4;
            _setUpPieces[6] = 4;
            _setUpPieces[7] = 3;
            _setUpPieces[8] = 2;
            _setUpPieces[9] = 1;
            _setUpPieces[10] = 1;
            _setUpPieces[11] = 1;
        }
        _boardPanel.Invalidate();
        _graveyardPanel.Invalidate();
    }
}