/*
* Program: BDS.c
*
* Description: A server that receives commands from a client and then mimicks a physical disk.
*
* "BDS"
*
* To execute: 
*
* From command line type: ./BDS <filename> <cylinders> <sectors> <tracktotracksleep(microseconds)>
* where <filename> is the name of the file that you want to open
* where <cylinders> is the number of cylinders in the "physical disk"
* where <sectors> is the number of sectors per cylinder
* where <tracktotracksleep(microseconds)> is the amount of sleep time between changing cylinders
*
*
* Written by: Chuck Laderer
* Date last edited: November 1, 2009
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sstream>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <sys/stat.h>

#define SIZE 1024
#define TIME_PORT 50152
int BLOCKSIZE = 256;

//port ranges: 50150-50159

int main(int argc, char *argv[])
{
	//if there are not 5 arguments
	if(argc != 5)
	{
		printf("Error: Not enough arguments\n");
		exit(-1);
	}

	char *fileName = argv[1];
	int numCylinders = atoi(argv[2]), numSectors = atoi(argv[3]), sleepTime = atoi(argv[4]), c1=0, sockfd, client_sockfd, nread;
	socklen_t len;

	//opens a file
	int fd = open(fileName, O_RDWR | O_CREAT, 0);
	if(fd < 0)
	{
		fprintf(stderr, "Error: Could not open file '%s'.\n", fileName);
		exit(-1);
	}

	//stretches the file size to the size of the simulated disk
	long FILESIZE = BLOCKSIZE * numSectors * numCylinders;
	int result = lseek(fd, FILESIZE-1, SEEK_SET);
	if(result == -1)
	{
		fprintf(stderr, "Error calling lseek() to 'stretch' the file");
		close(fd);
		exit(-1);
	}

	//write something at the end of the file to ensure the file actually has the new size
	result = write(fd, "", 1);
	if(result != 1)
	{
		fprintf(stderr, "Error writing last byte of the file");
		close(fd);
		exit(-1);
	}

	//map the file
	char* diskfile;
	diskfile = (char *)mmap(0, FILESIZE, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
	if(diskfile == MAP_FAILED)
	{
		close(fd);
		fprintf(stderr, "Error: Could not map file.\n");
		exit(-1);
	}

	struct sockaddr_in serv_addr, client_addr;

	//create endpoint
	if((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		fprintf(stderr, "Error opening socket\n");
		exit(2);
	}

	//bind address
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	serv_addr.sin_port = htons(TIME_PORT);

	if(bind(sockfd, (sockaddr*)&serv_addr, sizeof(serv_addr)) < 0)
	{
		fprintf(stderr, "Error binding socket\n");
		exit(3);
	}

	//specify queue
	listen(sockfd, 5);
	for(;;)
	{
		len = sizeof(client_addr);
		client_sockfd = accept(sockfd, (sockaddr*)&client_addr, &len);
		if(client_sockfd == -1)
		{
			fprintf(stderr, "Error accepting connection\n");
			continue;
		}
		bool clientRunning=true;

		//while the client is still entering commands
		while(clientRunning == true)
		{
			char buf[SIZE], delims[] = " ";
			char *data = {0};
			int c, s, l;
			//read client string
			read(client_sockfd, buf, SIZE);

			//if the client entered 'exit'
			if(strcmp(buf, "exit") == 0)
			{
				close(client_sockfd);
				clientRunning = false;
			}
			//if the client entered any other command
			else
			{
				char dummyBuf[SIZE] = {0};
				//if the command from the user is potentially a valid one
				if(buf[0] == 'I' || buf[0] == 'W' || buf[0] == 'R')
				{
					//if the client enters an information command
					if(buf[0] == 'I')
					{
						//add the numCylinders and numSectors to the dummyBuf and write it to the client
						strcpy(dummyBuf, argv[2]);
						strcat(dummyBuf, " ");
						strcat(dummyBuf, argv[3]);

						write(client_sockfd, dummyBuf, (strlen(dummyBuf) + 1));
					}
					//if it could be the read or write command
					else
					{
						//get the command, cylinder, and sector from the entered command
						strtok(buf, delims);
						c = atoi(strtok(NULL, delims));
						s = atoi(strtok(NULL, delims));

						//if the cylidner requested is invalid
						if(c >= numCylinders || c < 0)
						{
							dummyBuf[0] = '0';
							write(client_sockfd, dummyBuf, (strlen(dummyBuf) + 1));
						}
						//if the cylinder request is valid
						else
						{
							//if the sector requested is invalid
							if(s >= numSectors || s < 0)
							{
								dummyBuf[0] = '0';
								write(client_sockfd, dummyBuf, (strlen(dummyBuf) + 1));
							}
							//if sector request is valid
							else
							{
								//if the client enters a read command
								if(buf[0] == 'R')
								{
									char dataBuf[SIZE] = {0};

									strcat(dummyBuf, "1");

									//if the cylinder is not in the requested cylinder
									if(c != c1)
									{
										usleep(sleepTime * abs(c - c1));
										c1 = c;
									}

									//read from the "disk"
									memcpy(dataBuf, &diskfile[BLOCKSIZE * (c * numSectors + s)], BLOCKSIZE);

									strcat(dummyBuf, dataBuf);

									write(client_sockfd, dummyBuf, (strlen(dummyBuf) + 1));
								}
								//if the client enters a write command
								else if(buf[0] == 'W')
								{
									//get the length and data from the user command
									l = atoi(strtok(NULL, delims));
									data = strtok(NULL, delims);

									//if the sector requested is invalid
									if(l < 0)
									{
										dummyBuf[0] = '0';
										write(client_sockfd, dummyBuf, (strlen(dummyBuf) + 1));
									}
									//if the byte size is valid
									else
									{
										char nullString[] = {'\0'};
										for(int x=l; x<BLOCKSIZE+1; x++)
										{
											strcat(data, nullString);
										}

										//if the cylinder is not in the requested cylinder
										if(c != c1)
										{
											usleep(sleepTime * abs(c - c1));
											c1 = c;
										}

										//write to the "disk"
										memcpy(&diskfile[BLOCKSIZE * (c * numSectors + s)], data, BLOCKSIZE);

										strcat(dummyBuf, "1");

										write(client_sockfd, dummyBuf, (strlen(dummyBuf) + 1));
									}
								}
							}
						}
					}
				}
				//if the command is not potentially valid
				else
				{
					fprintf(stderr, "Invalid server command\n");
					write(client_sockfd, buf, (strlen(buf) + 1));
				}
			}
		}
		fprintf(stderr, "Server disconnected client\n");
	}
}
