/*
* Program: IDSS.c
*
* Description: A server that receives commands from a client and then mimicks a physical disk.
*
* "IDSS"
*
* To execute: 
*
* From command line type: ./IDSS <filename> <cylinders> <sectors> <tracktotracksleep(microseconds)> <seed>
* where <filename> is the name of the file that you want to open
* where <cylinders> is the number of cylinders in the "physical disk"
* where <sectors> is the number of sectors per cylinder
* where <tracktotracksleep(microseconds)> is the amount of sleep time between changing cylinders
* where <seed> is the number used for the random number generator
*
* Written by: Chuck Laderer
* Date last edited: November 16, 2009
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sstream>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <sys/stat.h>

#define SIZE 1024
#define TIME_PORT 50153
int BLOCKSIZE = 256;

//port ranges: 50150-50159

int main(int argc, char *argv[])
{
	//if there are not 5 arguments
	if(argc != 5)
	{
		printf("Error: Not enough arguments\n");
		exit(-1);
	}

	char *fileName = argv[1];
	int numCylinders = atoi(argv[2]), numSectors = atoi(argv[3]), sleepTime = atoi(argv[4]), c1=0, sockfd, client_sockfd, nread;
	socklen_t len;

	//opens a file
	int fd = open(fileName, O_RDWR | O_CREAT, 0);
	if(fd < 0)
	{
		fprintf(stderr, "Error: Could not open file '%s'.\n", fileName);
		exit(-1);
	}

	//stretches the file size to the size of the simulated disk
	long FILESIZE = BLOCKSIZE * numSectors * numCylinders;
	int result = lseek(fd, FILESIZE-1, SEEK_SET);
	if(result == -1)
	{
		fprintf(stderr, "Error calling lseek() to 'stretch' the file");
		close(fd);
		exit(-1);
	}

	//write something at the end of the file to ensure the file actually has the new size
	result = write(fd, "", 1);
	if(result != 1)
	{
		fprintf(stderr, "Error writing last byte of the file");
		close(fd);
		exit(-1);
	}

	//map the file
	char* diskfile;
	diskfile = (char *)mmap(0, FILESIZE, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0);
	if(diskfile == MAP_FAILED)
	{
		close(fd);
		fprintf(stderr, "Error: Could not map file.\n");
		exit(-1);
	}

	struct sockaddr_in serv_addr, client_addr;

	//create endpoint
	if((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	{
		fprintf(stderr, "Error opening socket\n");
		exit(2);
	}

	//bind address
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_addr.s_addr = htonl(INADDR_ANY);
	serv_addr.sin_port = htons(TIME_PORT);

	if(bind(sockfd, (sockaddr*)&serv_addr, sizeof(serv_addr)) < 0)
	{
		fprintf(stderr, "Error binding socket\n");
		exit(3);
	}

	//specify queue
	listen(sockfd, 5);
	for(;;)
	{
		len = sizeof(client_addr);
		client_sockfd = accept(sockfd, (sockaddr*)&client_addr, &len);
		if(client_sockfd == -1)
		{
			fprintf(stderr, "Error accepting connection\n");
			continue;
		}
		bool clientRunning=true;

		char buf[SIZE] = {0}, dummyBuf[SIZE] = {0};

		read(client_sockfd, buf, SIZE);

		//add the numCylinders and numSectors to the dummyBuf and write it to the client
		strcpy(dummyBuf, argv[2]);
		strcat(dummyBuf, " ");
		strcat(dummyBuf, argv[3]);

		write(client_sockfd, dummyBuf, (strlen(dummyBuf) + 1));

		//clears the buf and dummyBuf
		memset(buf, '\0', sizeof(buf));
		memset(dummyBuf, '\0', sizeof(dummyBuf));

		//reads the client entered input
		read(client_sockfd, buf, SIZE);

		int numRequests = atoi(strtok(buf, " ")), scheduler = atoi(strtok(NULL, " ")), bucketSize = atoi(strtok(NULL, " "));

		fprintf(stderr, "Num Requests: %i, Scheduler: %i, Bucketsize: %i\n", numRequests, scheduler, bucketSize);

		char output[SIZE] = {0};

		//reads all of the client commands
		read(client_sockfd, buf, SIZE);

		int totalSleepTime = 0;
		//if they want the FCFS scheduler
		if(scheduler == 0)
		{
			char *command = {0};
			command = strtok(buf, " ");

			//loops through the data
			for(int x=0; x<numRequests; x++)
			{
				int c, s, l;
				//get the command, cylinder, and sector from the entered command
				c = atoi(strtok(NULL, " "));
				s = atoi(strtok(NULL, " "));

				//if the client enters a read command
				if(command[0] == 'R')
				{
					char dataBuf[SIZE] = {0};

					//if the current cylinder is not the requested cylinder
					if(c != c1)
					{
						int timeSleep = sleepTime * abs(c - c1);
						totalSleepTime += timeSleep;
						usleep(timeSleep);
						c1 = c;
					}
					//read from the "disk"
					memcpy(dataBuf, &diskfile[BLOCKSIZE * (c * numSectors + s)], BLOCKSIZE);
					strcat(output, "1");
				}
				//if the client enters a write command
				else if(command[0] == 'W')
				{
					char *data = {0};
					//get the length and data from the user command
					l = atoi(strtok(NULL, " "));
					data = strtok(NULL, " ");

					//if the current cylinder is not the requested cylinder
					if(c != c1)
					{
						int timeSleep = sleepTime * abs(c - c1);
						totalSleepTime += timeSleep;
						usleep(timeSleep);
						c1 = c;
					}
					//write to the "disk"
					memcpy(&diskfile[BLOCKSIZE * (c * numSectors + s)], data, BLOCKSIZE);
					strcat(output, "1");
				}
				command = strtok(NULL, " ");
			}
		}
		//if they want the SSTF scheduler
		else if(scheduler == 1)
		{
			int leftOver = (numRequests%bucketSize);

			char *command[SIZE] = {0};
			command[0] = strtok(buf, " ");

			//loops through the data in blocks of bucketSize
			for(int x=0; x<(numRequests/bucketSize); x++)
			{
				int c[SIZE] = {0}, s[SIZE] = {0}, l[SIZE] = {0};
				char *data[SIZE] = {0};
				//loops through the data in groups of bucketSize
				for(int y=0; y<bucketSize; y++)
				{
					//if it is not the first part of the data
					if(y != 0)
					{
						command[y] = strtok(NULL, " ");
					}
					//get the command, cylinder, and sector from the entered command
					c[y] = atoi(strtok(NULL, " "));
					s[y] = atoi(strtok(NULL, " "));
					//if the command is a write command
					if(command[y][0] == 'W')
					{
						//get the length and data from the user command
						l[y] = atoi(strtok(NULL, " "));
						data[y] = strtok(NULL, " ");
					}
				}

				//loop through the cylinders to change them and process commands
				for(int y=0; y<numCylinders; y++)
				{
					//loops through the groups of bucketSize
					for(int z=0; z<bucketSize; z++)
					{
						//if the command is in this cylinder
						if(c[z] == y)
						{
							//if the client enters a read command
							if(command[z][0] == 'R')
							{
								char dataBuf[SIZE] = {0};

								//if the current cylinder is not the requested cylinder
								if(c[z] != c1)
								{
									int timeSleep = sleepTime * abs(c[z] - c1);
									totalSleepTime += timeSleep;
									usleep(timeSleep);
									c1 = c[z];
								}
								//read from the "disk"
								memcpy(dataBuf, &diskfile[BLOCKSIZE * (c[z] * numSectors + s[z])], BLOCKSIZE);
								strcat(output, "1");
							}
							//if the client enters a write command
							else if(command[z][0] == 'W')
							{
								//if the current cylinder is not the requested cylinder
								if(c[z] != c1)
								{
									int timeSleep = sleepTime * abs(c[z] - c1);
									totalSleepTime += timeSleep;
									usleep(timeSleep);
									c1 = c[z];
								}								
								//write to the "disk"
								memcpy(&diskfile[BLOCKSIZE * (c[z] * numSectors + s[z])], data[z], BLOCKSIZE);
								strcat(output, "1");
							}
						}
					}
				}
				//resets the cylinder and command[0]
				c1 = 0;
				command[0] = strtok(NULL, " ");
			}

			int c[SIZE] = {0}, s[SIZE] = {0}, l[SIZE] = {0};
			char *data[SIZE] = {0};
			//loops through the leftOver commands
			for(int y=0; y<leftOver; y++)
			{
				//if it is not the first part of the data
				if(y != 0)
				{
					command[y] = strtok(NULL, " ");
				}
				//get the command, cylinder, and sector from the entered command
				c[y] = atoi(strtok(NULL, " "));
				s[y] = atoi(strtok(NULL, " "));
				//if the command is a write command
				if(command[y][0] == 'W')
				{
					//get the length and data from the entered command
					l[y] = atoi(strtok(NULL, " "));
					data[y] = strtok(NULL, " ");
				}
			}

			//loop through the cylinders to change them and process commands
			for(int y=0; y<numCylinders; y++)
			{
				//loops through the leftover commands
				for(int z=0; z<leftOver; z++)
				{
					//if the command is in this cylinder
					if(c[z] == y)
					{
						//if the client enters a read command
						if(command[z][0] == 'R')
						{
							char dataBuf[SIZE] = {0};

							//if the current cylinder is not the requested cylinder
							if(c[z] != c1)
							{
								int timeSleep = sleepTime * abs(c[z] - c1);
								totalSleepTime += timeSleep;
								usleep(timeSleep);
								c1 = c[z];
							}
							//read from the "disk"
							memcpy(dataBuf, &diskfile[BLOCKSIZE * (c[z] * numSectors + s[z])], BLOCKSIZE);
							strcat(output, "1");
						}
						//if the client enters a write command
						else if(command[z][0] == 'W')
						{
							//if the current cylinder is not the requested cylinder
							if(c[z] != c1)
							{
								int timeSleep = sleepTime * abs(c[z] - c1);
								totalSleepTime += timeSleep;
								usleep(timeSleep);
								c1 = c[z];
							}								
							//write to the "disk"
							memcpy(&diskfile[BLOCKSIZE * (c[z] * numSectors + s[z])], data[z], BLOCKSIZE);
							strcat(output, "1");
						}
					}
				}
			}
		}
		//if they want the C-LOOK scheduler
		else if(scheduler == 2)
		{
			char *command[SIZE] = {0}, *data[SIZE] = {0};
			int c[SIZE] = {0}, s[SIZE] = {0}, l[SIZE] = {0};

			command[0] = strtok(buf, " ");

			//loops until all the requests have been processed
			for(int y=0; y<numRequests; y++)
			{
				//if it is not the first command
				if(y != 0)
				{
					command[y] = strtok(NULL, " ");
				}
				//get the command, cylinder, and sector from the entered command
				c[y] = atoi(strtok(NULL, " "));
				s[y] = atoi(strtok(NULL, " "));
				if(command[y][0] == 'W')
				{
					//get the length and data from the entered command
					l[y] = atoi(strtok(NULL, " "));
					data[y] = strtok(NULL, " ");
				}
			}

			//loop through the cylinders to change them and process commands
			for(int y=0; y<numCylinders; y++)
			{
				//loop through all the commands
				for(int z=0; z<numRequests; z++)
				{
					//if the command is in this cylinder
					if(c[z] == y)
					{
						//if the client enters a read command
						if(command[z][0] == 'R')
						{
							char dataBuf[SIZE] = {0};

							//if the current cylinder is not the requested cylinder
							if(c[z] != c1)
							{
								int timeSleep = sleepTime * abs(c[z] - c1);
								totalSleepTime += timeSleep;
								usleep(timeSleep);
								c1 = c[z];
							}
							//read from the "disk"
							memcpy(dataBuf, &diskfile[BLOCKSIZE * (c[z] * numSectors + s[z])], BLOCKSIZE);
							strcat(output, "1");
						}
						//if the client enters a write command
						else if(command[z][0] == 'W')
						{
							//if the current cylinder is not the requested cylinder
							if(c[z] != c1)
							{
								int timeSleep = sleepTime * abs(c[z] - c1);
								totalSleepTime += timeSleep;
								usleep(timeSleep);
								c1 = c[z];
							}								
							//write to the "disk"
							memcpy(&diskfile[BLOCKSIZE * (c[z] * numSectors + s[z])], data[z], BLOCKSIZE);
							strcat(output, "1");
						}
					}
				}
			}
		}

		write(client_sockfd, output, (strlen(output) + 1));

		fprintf(stderr, "Client Disconnected. Total Sleep Time is: %i\n", totalSleepTime);
	}
}
